//-----------------------------------------------------------------------------
// Torque Game Engine
// Copyright (C) GarageGames.com, Inc.
//-----------------------------------------------------------------------------

#include "core/strings/stringFunctions.h"
#include "core/strings/unicode.h"

#include "math/frustum.h"
#include "math/util/sphereMesh.h"

#include "gfx/gfxDrawUtil.h"
#include "gfx/gfxFontRenderBatcher.h"


GFXDrawUtil::GFXDrawUtil( GFXDevice * d)
{
   mDevice = d;
   mBitmapModulation.set(0xFF, 0xFF, 0xFF, 0xFF);
   mTextAnchorColor.set(0xFF, 0xFF, 0xFF, 0xFF);
   mFontRenderBatcher = new FontRenderBatcher();
   setupStateBlocks();
}

GFXDrawUtil::~GFXDrawUtil()
{
   delete mFontRenderBatcher;
}


void GFXDrawUtil::setupStateBlocks()
{
   // DrawBitmapStretchSR
   GFXStateBlockDesc bitmapStretchSR;
   bitmapStretchSR.setCullMode(GFXCullNone);
   bitmapStretchSR.setZEnable(false);
   bitmapStretchSR.setBlend(true, GFXBlendSrcAlpha, GFXBlendInvSrcAlpha);
   bitmapStretchSR.samplersDefined = true;

   // Linear: Create wrap SB
   bitmapStretchSR.samplers[0] = GFXSamplerStateDesc::getWrapLinear();
   mBitmapStretchWrapLinearSB = mDevice->createStateBlock(bitmapStretchSR);

   // Linear: Create clamp SB
   bitmapStretchSR.samplers[0] = GFXSamplerStateDesc::getClampLinear();
   mBitmapStretchLinearSB = mDevice->createStateBlock(bitmapStretchSR);

   // Point:
   bitmapStretchSR.samplers[0].minFilter = GFXTextureFilterPoint;
   bitmapStretchSR.samplers[0].mipFilter = GFXTextureFilterPoint;
   bitmapStretchSR.samplers[0].magFilter = GFXTextureFilterPoint;

   // Point: Create clamp SB, last created clamped so no work required here
   mBitmapStretchSB = mDevice->createStateBlock(bitmapStretchSR);

   // Point: Create wrap SB, have to do this manually because getWrapLinear doesn't
   bitmapStretchSR.samplers[0].addressModeU = GFXAddressWrap;
   bitmapStretchSR.samplers[0].addressModeV = GFXAddressWrap;
   bitmapStretchSR.samplers[0].addressModeW = GFXAddressWrap;
   mBitmapStretchWrapSB = mDevice->createStateBlock(bitmapStretchSR);

   GFXStateBlockDesc rectFill;
   rectFill.setCullMode(GFXCullNone);
   rectFill.setZEnable(false);
   rectFill.setBlend(true, GFXBlendSrcAlpha, GFXBlendInvSrcAlpha);
   mRectFillSB = mDevice->createStateBlock(rectFill);

   GFXStateBlockDesc solid;
   solid.setCullMode( GFXCullNone );
   mSolidSB = mDevice->createStateBlock( solid );

   GFXStateBlockDesc transparent;
   transparent.setCullMode(GFXCullNone);
   transparent.setZEnable(true);
   transparent.zWriteEnable = false;
   transparent.setBlend(true, GFXBlendSrcAlpha, GFXBlendInvSrcAlpha);
   mTrasparentSB = mDevice->createStateBlock( transparent );
}

//-----------------------------------------------------------------------------
// Color Modulation
//-----------------------------------------------------------------------------
void GFXDrawUtil::setBitmapModulation( const ColorI &modColor )
{
   mBitmapModulation = modColor;
}

void GFXDrawUtil::clearBitmapModulation()
{
   mBitmapModulation.set( 255, 255, 255, 255 );
}

void GFXDrawUtil::getBitmapModulation( ColorI *color )
{
   mBitmapModulation.getColor( color );
}

void GFXDrawUtil::setTextAnchorColor( const ColorI &ancColor )
{
   mTextAnchorColor = ancColor;
}

//-----------------------------------------------------------------------------
// Draw Text
//-----------------------------------------------------------------------------
U32 GFXDrawUtil::drawText( GFont *font, const Point2I &ptDraw, const UTF16 *in_string, 
                          const ColorI *colorTable, const U32 maxColorIndex, F32 rot )
{
   return drawTextN( font, ptDraw, in_string, dStrlen(in_string), colorTable, maxColorIndex, rot );
}

U32 GFXDrawUtil::drawText( GFont *font, const Point2I &ptDraw, const UTF8 *in_string, 
                          const ColorI *colorTable, const U32 maxColorIndex, F32 rot )
{
   return drawTextN( font, ptDraw, in_string, dStrlen(in_string), colorTable, maxColorIndex, rot );
}

U32 GFXDrawUtil::drawText( GFont *font, const Point2F &ptDraw, const UTF8 *in_string, const ColorI *colorTable /*= NULL*/, const U32 maxColorIndex /*= 9*/, F32 rot /*= 0.f */ )
{
   return drawText(font,Point2I((S32)ptDraw.x,(S32)ptDraw.y),in_string,colorTable,maxColorIndex,rot);
}

U32 GFXDrawUtil::drawText( GFont *font, const Point2F &ptDraw, const UTF16 *in_string, const ColorI *colorTable /*= NULL*/, const U32 maxColorIndex /*= 9*/, F32 rot /*= 0.f */ )
{
   return drawText(font,Point2I((S32)ptDraw.x,(S32)ptDraw.y),in_string,colorTable,maxColorIndex,rot);
}

U32 GFXDrawUtil::drawTextN( GFont *font, const Point2I &ptDraw, const UTF8 *in_string, U32 n,
                           const ColorI *colorTable, const U32 maxColorIndex, F32 rot )
{
   // return on zero length strings
   if( n == 0 )
      return ptDraw.x;

   // Convert to UTF16 temporarily.
   n++; // space for null terminator
   FrameTemp<UTF16> ubuf( n * sizeof(UTF16) );
   convertUTF8toUTF16(in_string, ubuf, n);

   return drawTextN( font, ptDraw, ubuf, n, colorTable, maxColorIndex, rot );
}

U32 GFXDrawUtil::drawTextN( GFont *font, const Point2I &ptDraw, const UTF16 *in_string, 
                           U32 n, const ColorI *colorTable, const U32 maxColorIndex, F32 rot )
{
   // return on zero length strings
   if( n == 0 )
      return ptDraw.x;

   // If it's over about 4000 verts we want to break it up
   if( n > 666 )
   {
      U32 left = drawTextN(font, ptDraw, in_string, 666, colorTable, maxColorIndex, rot);

      Point2I newDrawPt(left, ptDraw.y);
      const UTF16* str = (const UTF16*)in_string;

      return drawTextN(font, newDrawPt, &(str[666]), n - 666, colorTable, maxColorIndex, rot);
   }

   PROFILE_START(GFXDevice_drawTextN);

   const PlatformFont::CharInfo *tabci = NULL;

   S32 ptX = 0;

   // Queue everything for render.   
   mFontRenderBatcher->init(font, n);

   U32 i;
   UTF16 c;   
   for(i = 0, c = in_string[i]; in_string[i] && i < n; i++, c = in_string[i])
   {
      switch(c)
      {
         // We have to do a little dance here since \t = 0x9, \n = 0xa, and \r = 0xd
      case 1: case 2: case 3: case 4: case 5: case 6: case 7:
      case 11: case 12:
      case 14:
         {
            // Color code
            if (colorTable) 
            {
               static U8 remap[15] = 
               { 
                  0x0, // 0 special null terminator
                  0x0, // 1 ascii start-of-heading??
                  0x1, 
                  0x2, 
                  0x3, 
                  0x4, 
                  0x5, 
                  0x6, 
                  0x0, // 8 special backspace
                  0x0, // 9 special tab
                  0x0, // a special \n
                  0x7, 
                  0x8,
                  0x0, // a special \r
                  0x9 
               };

               U8 remapped = remap[c];

               // Ignore if the color is greater than the specified max index:
               if ( remapped <= maxColorIndex )
               {
                  const ColorI &clr = colorTable[remapped];
                  mBitmapModulation = clr;
               }
            }

            // And skip rendering this character.
            continue;
         }

         // reset color?
      case 15:
         {
            mBitmapModulation = mTextAnchorColor;

            // And skip rendering this character.
            continue;
         }

         // push color:
      case 16:
         {
            mTextAnchorColor = mBitmapModulation;

            // And skip rendering this character.
            continue;
         }

         // pop color:
      case 17:
         {
            mBitmapModulation = mTextAnchorColor;

            // And skip rendering this character.
            continue;
         }

         // Tab character
      case dT('\t'): 
         {
            if ( tabci == NULL )
               tabci = &(font->getCharInfo( dT(' ') ));

            const U32	fontTabIncrement = tabci->xIncrement * GFont::TabWidthInSpaces;

            ptX += fontTabIncrement;

            // And skip rendering this character.
            continue;
         }

         // Don't draw invalid characters.
      default:
         {
            if( !font->isValidChar( c ) ) 
               continue;
         }
      }

      // Queue char for rendering..
      mFontRenderBatcher->queueChar(c, ptX, mBitmapModulation);
   }


   mFontRenderBatcher->render(rot, Point2F((F32)ptDraw.x, (F32)ptDraw.y));

   PROFILE_END();

   return ptX + ptDraw.x;
}

U32 GFXDrawUtil::drawTextN( GFont *font, const Point2F &ptDraw, const UTF8 *in_string, U32 n, const ColorI *colorTable /*= NULL*/, const U32 maxColorIndex /*= 9*/, F32 rot /*= 0.f */ )
{
   return drawTextN(font,Point2I((S32)ptDraw.x,(S32)ptDraw.y),in_string,n,colorTable,maxColorIndex,rot);
}

U32 GFXDrawUtil::drawTextN( GFont *font, const Point2F &ptDraw, const UTF16 *in_string, U32 n, const ColorI *colorTable /*= NULL*/, const U32 maxColorIndex /*= 9*/, F32 rot /*= 0.f */ )
{
   return drawTextN(font,Point2I((S32)ptDraw.x,(S32)ptDraw.y),in_string,n,colorTable,maxColorIndex,rot);
}

//-----------------------------------------------------------------------------
// Draw Bitmaps
//-----------------------------------------------------------------------------
void GFXDrawUtil::drawBitmap( GFXTextureObject* texture, const Point2I &in_rAt, const GFXBitmapFlip in_flip, const GFXTextureFilterType filter , bool in_wrap /*= true*/ )
{
   drawBitmap(texture,Point2F((F32)in_rAt.x,(F32)in_rAt.y),in_flip,filter,in_wrap);
}

void GFXDrawUtil::drawBitmapStretch( GFXTextureObject* texture, const RectI &dstRect, const GFXBitmapFlip in_flip, const GFXTextureFilterType filter , bool in_wrap /*= true*/ )
{
   drawBitmapStretch(texture,RectF((F32)dstRect.point.x,(F32)dstRect.point.y,(F32)dstRect.extent.x,(F32)dstRect.extent.y),in_flip,filter,in_wrap);
}

void GFXDrawUtil::drawBitmapSR( GFXTextureObject* texture, const Point2I &in_rAt, const RectI &srcRect, const GFXBitmapFlip in_flip, const GFXTextureFilterType filter , bool in_wrap /*= true*/ )
{
   drawBitmapSR(texture,Point2F((F32)in_rAt.x,(F32)in_rAt.y),RectF((F32)srcRect.point.x,(F32)srcRect.point.y,(F32)srcRect.extent.x,(F32)srcRect.extent.y),in_flip,filter,in_wrap);
}

void GFXDrawUtil::drawBitmapStretchSR( GFXTextureObject *texture, const RectI &dstRect, const RectI &srcRect, const GFXBitmapFlip in_flip, const GFXTextureFilterType filter , bool in_wrap /*= true*/ ) 
{
   RectF dstRectF = RectF((F32)dstRect.point.x,(F32)dstRect.point.y,(F32)dstRect.extent.x,(F32)dstRect.extent.y);
   RectF srcRectF = RectF((F32)srcRect.point.x,(F32)srcRect.point.y,(F32)srcRect.extent.x,(F32)srcRect.extent.y);
   drawBitmapStretchSR(texture,dstRectF,srcRectF,in_flip,filter,in_wrap);
}

void GFXDrawUtil::drawBitmap( GFXTextureObject*texture, const Point2F &in_rAt, const GFXBitmapFlip in_flip /*= GFXBitmapFlip_None*/, const GFXTextureFilterType filter /*= GFXTextureFilterPoint */ , bool in_wrap /*= true*/ )
{
   AssertFatal( texture != 0, "No texture specified for drawBitmap()" );

   RectI subRegion( 0, 0, texture->mBitmapSize.x, texture->mBitmapSize.y );
   RectI stretch( in_rAt.x, in_rAt.y, texture->mBitmapSize.x, texture->mBitmapSize.y );
   drawBitmapStretchSR( texture, stretch, subRegion, in_flip, filter, in_wrap );
}

void GFXDrawUtil::drawBitmapStretch( GFXTextureObject*texture, const RectF &dstRect, const GFXBitmapFlip in_flip /*= GFXBitmapFlip_None*/, const GFXTextureFilterType filter /*= GFXTextureFilterPoint */ , bool in_wrap /*= true*/ )
{
   AssertFatal( texture != 0, "No texture specified for drawBitmapStretch()" );

   RectF subRegion( 0.f, 0.f, (F32)texture->mBitmapSize.x, (F32)texture->mBitmapSize.y );
   drawBitmapStretchSR( texture, dstRect, subRegion, in_flip, filter, in_wrap );
}

void GFXDrawUtil::drawBitmapSR( GFXTextureObject*texture, const Point2F &in_rAt, const RectF &srcRect, const GFXBitmapFlip in_flip /*= GFXBitmapFlip_None*/, const GFXTextureFilterType filter /*= GFXTextureFilterPoint */ , bool in_wrap /*= true*/ )
{
   AssertFatal( texture != 0, "No texture specified for drawBitmapSR()" );

   RectF stretch( in_rAt.x, in_rAt.y, srcRect.len_x(), srcRect.len_y() );
   drawBitmapStretchSR( texture, stretch, srcRect, in_flip, filter, in_wrap );
}

void GFXDrawUtil::drawBitmapStretchSR( GFXTextureObject* texture, const RectF &dstRect, const RectF &srcRect, const GFXBitmapFlip in_flip /*= GFXBitmapFlip_None*/, const GFXTextureFilterType filter /*= GFXTextureFilterPoint */ , bool in_wrap /*= true*/ )
{
   // Sanity if no texture is specified.
   if(!texture)
      return;   

   GFXVertexBufferHandle<GFXVertexPCT> verts(mDevice, 4, GFXBufferTypeVolatile );
   verts.lock();

   F32 texLeft   = (srcRect.point.x)                    / (texture->mTextureSize.x);
   F32 texRight  = (srcRect.point.x + srcRect.extent.x) / (texture->mTextureSize.x);
   F32 texTop    = (srcRect.point.y)                    / (texture->mTextureSize.y);
   F32 texBottom = (srcRect.point.y + srcRect.extent.y) / (texture->mTextureSize.y);

   F32 screenLeft   = dstRect.point.x;
   F32 screenRight  = (dstRect.point.x + dstRect.extent.x);
   F32 screenTop    = dstRect.point.y;
   F32 screenBottom = (dstRect.point.y + dstRect.extent.y);

   if( in_flip & GFXBitmapFlip_X ) 
   {
      F32 temp = texLeft;
      texLeft = texRight;
      texRight = temp;
   }
   if( in_flip & GFXBitmapFlip_Y ) 
   {
      F32 temp = texTop;
      texTop = texBottom;
      texBottom = temp;
   }

   const F32 fillConv = mDevice->getFillConventionOffset();
   verts[0].point.set( screenLeft  - fillConv, screenTop    - fillConv, 0.f );
   verts[1].point.set( screenRight - fillConv, screenTop    - fillConv, 0.f );
   verts[2].point.set( screenLeft  - fillConv, screenBottom - fillConv, 0.f );
   verts[3].point.set( screenRight - fillConv, screenBottom - fillConv, 0.f );

   verts[0].color = verts[1].color = verts[2].color = verts[3].color = mBitmapModulation;

   verts[0].texCoord.set( texLeft,  texTop );
   verts[1].texCoord.set( texRight, texTop );
   verts[2].texCoord.set( texLeft,  texBottom );
   verts[3].texCoord.set( texRight, texBottom );

   verts.unlock();

   mDevice->setVertexBuffer( verts );

   switch (filter)
   {
   case GFXTextureFilterPoint :
      mDevice->setStateBlock(in_wrap ? mBitmapStretchWrapSB : mBitmapStretchSB);
      break;
   case GFXTextureFilterLinear :
      mDevice->setStateBlock(in_wrap ? mBitmapStretchWrapLinearSB : mBitmapStretchLinearSB);
      break;
   default:
      AssertFatal(false, "No GFXDrawUtil state block defined for this filter type!");
      mDevice->setStateBlock(mBitmapStretchSB);
      break;
   }   
   mDevice->setTexture( 0, texture );
   mDevice->setupGenericShaders( GFXDevice::GSModColorTexture );

   mDevice->drawPrimitive( GFXTriangleStrip, 0, 2 );
}

//-----------------------------------------------------------------------------
// 3D World Draw Misc
//-----------------------------------------------------------------------------
void GFXDrawUtil::draw2DSquare( const Point2F &screenPoint, F32 width, F32 spinAngle )
{
   width *= 0.5;

   Point3F offset( screenPoint.x, screenPoint.y, 0.0 );

   GFXVertexBufferHandle<GFXVertexPC> verts( mDevice, 4, GFXBufferTypeVolatile );
   verts.lock();

   verts[0].point.set( -width, -width, 0.0f );
   verts[1].point.set( -width, width, 0.0f );
   verts[2].point.set( width,  -width, 0.0f );
   verts[3].point.set( width,  width, 0.0f );

   verts[0].color = verts[1].color = verts[2].color = verts[3].color = mBitmapModulation;

   if(spinAngle != 0.f)
   {
      MatrixF rotMatrix( EulerF( 0.0, 0.0, spinAngle ) );

      for( S32 i = 0; i < 4; i++ )
      {
         rotMatrix.mulP( verts[i].point );
         verts[i].point += offset;
      }
   }

   verts.unlock();
   mDevice->setVertexBuffer( verts );

   mDevice->setStateBlock(mRectFillSB);
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleStrip, 0, 2 );
}

static SphereMesh gSphere;

void GFXDrawUtil::drawWireSphere( F32 radius, const Point3F & pos, const ColorI & color, bool drawTop, bool drawBottom)
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(Point3F(radius,radius,radius));
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   const SphereMesh::TriangleMesh * sphereMesh = gSphere.getMesh(2);
   S32 numPoly = sphereMesh->numPoly;
   S32 totalPoly = 0;
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoly*4, GFXBufferTypeVolatile);
   verts.lock();
   S32 vertexIndex = 0;
   for (S32 i=0; i<numPoly; i++)
   {
      if (!drawBottom)
      {
         if (sphereMesh->poly[i].pnt[0].z < -0.01f || sphereMesh->poly[i].pnt[1].z < -0.01f || sphereMesh->poly[i].pnt[2].z < -0.01f)
            continue;
      }
      if (!drawTop)
      {
         if (sphereMesh->poly[i].pnt[0].z > 0.01f || sphereMesh->poly[i].pnt[1].z > 0.01f || sphereMesh->poly[i].pnt[2].z > 0.01f)
            continue;
      }
      totalPoly++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[0];
      verts[vertexIndex].color = color;
      vertexIndex++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[1];
      verts[vertexIndex].color = color;
      vertexIndex++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[2];
      verts[vertexIndex].color = color;
      vertexIndex++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[0];
      verts[vertexIndex].color = color;
      vertexIndex++;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   for (S32 i=0; i<totalPoly; i++)
      mDevice->drawPrimitive( GFXLineStrip, i*4, 3);

   GFX->popWorldMatrix();
}

void GFXDrawUtil::drawSolidSphere( F32 radius, const Point3F & pos, const ColorI & color, bool drawTop, bool drawBottom)
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(Point3F(radius,radius,radius));
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   const SphereMesh::TriangleMesh * sphereMesh = gSphere.getMesh(2);
   S32 numPoly = sphereMesh->numPoly;
   S32 totalPoly = 0;
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoly*3, GFXBufferTypeVolatile);
   verts.lock();
   S32 vertexIndex = 0;
   for (S32 i=0; i<numPoly; i++)
   {
      if (!drawBottom)
      {
         if (sphereMesh->poly[i].pnt[0].z < -0.01f || sphereMesh->poly[i].pnt[1].z < -0.01f || sphereMesh->poly[i].pnt[2].z < -0.01f)
            continue;
      }
      if (!drawTop)
      {
         if (sphereMesh->poly[i].pnt[0].z > 0.01f || sphereMesh->poly[i].pnt[1].z > 0.01f || sphereMesh->poly[i].pnt[2].z > 0.01f)
            continue;
      }
      totalPoly++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[0];
      verts[vertexIndex].color = color;
      vertexIndex++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[1];
      verts[vertexIndex].color = color;
      vertexIndex++;

      verts[vertexIndex].point = sphereMesh->poly[i].pnt[2];
      verts[vertexIndex].color = color;
      vertexIndex++;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleList, 0, totalPoly );

   GFX->popWorldMatrix();
}

//-----------------------------------------------------------------------------

static const Point3F cubePoints[8] = 
{
   Point3F(-1, -1, -1), Point3F(-1, -1,  1), Point3F(-1,  1, -1), Point3F(-1,  1,  1),
   Point3F( 1, -1, -1), Point3F( 1, -1,  1), Point3F( 1,  1, -1), Point3F( 1,  1,  1)
};

static const U32 cubeFaces[6][4] = 
{
   { 0, 4, 6, 2 }, { 0, 2, 3, 1 }, { 0, 1, 5, 4 },
   { 3, 2, 6, 7 }, { 7, 6, 4, 5 }, { 3, 7, 5, 1 }
};

void GFXDrawUtil::drawWireBox( const Box3F &box, const ColorI &color )
{
   Point3F  size = (box.maxExtents - box.minExtents) * 0.5f;

   drawWireCube( size, box.getCenter(), color );
}

void GFXDrawUtil::drawSolidBox( const Box3F &box, const ColorI &color )
{
   Point3F  size = (box.maxExtents - box.minExtents) * 0.5f;

   drawSolidCube( size, box.getCenter(), color );
}

void GFXDrawUtil::drawWireCube( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, 30, GFXBufferTypeVolatile);
   verts.lock();

   // setup 6 line loops
   U32 vertexIndex = 0;
   for(int i = 0; i < 6; i++)
   {
      for(int j = 0; j < 5; j++)
      {
         int idx = cubeFaces[i][j%4];

         verts[vertexIndex].point = cubePoints[idx] * size + pos;
         verts[vertexIndex].color = color;
         vertexIndex++;
      }
   }

   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   for( U32 i=0; i<6; i++ )
      mDevice->drawPrimitive( GFXLineStrip, i*5, 4 );
}

void GFXDrawUtil::drawSolidCube( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, 36, GFXBufferTypeVolatile);
   verts.lock();

   // setup 6 line loops
   U32 vertexIndex = 0;
   U32 idx;
   for(int i = 0; i < 6; i++)
   {
      idx = cubeFaces[i][0];
      verts[vertexIndex].point = cubePoints[idx] * size + pos;
      verts[vertexIndex].color = color;
      vertexIndex++;

      idx = cubeFaces[i][1];
      verts[vertexIndex].point = cubePoints[idx] * size + pos;
      verts[vertexIndex].color = color;
      vertexIndex++;

      idx = cubeFaces[i][3];
      verts[vertexIndex].point = cubePoints[idx] * size + pos;
      verts[vertexIndex].color = color;
      vertexIndex++;

      idx = cubeFaces[i][1];
      verts[vertexIndex].point = cubePoints[idx] * size + pos;
      verts[vertexIndex].color = color;
      vertexIndex++;

      idx = cubeFaces[i][3];
      verts[vertexIndex].point = cubePoints[idx] * size + pos;
      verts[vertexIndex].color = color;
      vertexIndex++;

      idx = cubeFaces[i][2];
      verts[vertexIndex].point = cubePoints[idx] * size + pos;
      verts[vertexIndex].color = color;
      vertexIndex++;
   }

   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleList, 0, 12 );
}

//-----------------------------------------------------------------------------
// Draw cylinder
//-----------------------------------------------------------------------------

static const Point2F circlePoints[] = 
{
   Point2F(0.707107f, 0.707107f),
   Point2F(0.923880f, 0.382683f),
   Point2F(1.000000f, 0.000000f),
   Point2F(0.923880f, -0.382684f),
   Point2F(0.707107f, -0.707107f),
   Point2F(0.382683f, -0.923880f),
   Point2F(0.000000f, -1.000000f),
   Point2F(-0.382683f, -0.923880f),
   Point2F(-0.707107f, -0.707107f),
   Point2F(-0.923880f, -0.382684f),
   Point2F(-1.000000f, 0.000000f),
   Point2F(-0.923879f, 0.382684f),
   Point2F(-0.707107f, 0.707107f),
   Point2F(-0.382683f, 0.923880f),
   Point2F(0.000000f, 1.000000f),
   Point2F(0.382684f, 0.923879f)
};

void GFXDrawUtil::drawSolidCylinder( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(0.5f * size);
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   S32 numPoints = sizeof(circlePoints)/sizeof(Point2F);
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoints * 4 + 4, GFXBufferTypeVolatile);
   verts.lock();
   for (S32 i=0; i<numPoints + 1; i++)
   {
      S32 imod = i % numPoints;
      verts[i].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, 1.0f);
      verts[i].color = color;
      verts[i + numPoints + 1].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, -1.0f);
      verts[i + numPoints + 1].color = color;

      verts[2*numPoints + 2 + 2 * i].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, 1.0f);
      verts[2*numPoints + 2 + 2 * i].color = color;
      verts[2*numPoints + 2 + 2 * i + 1].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, -1.0f);
      verts[2*numPoints + 2 + 2 * i + 1].color = color;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleFan, 0, numPoints );
   mDevice->drawPrimitive( GFXTriangleFan, numPoints + 1, numPoints );
   mDevice->drawPrimitive( GFXTriangleStrip, 2 * numPoints + 2, 2 * numPoints);

   GFX->popWorldMatrix();
}

void GFXDrawUtil::drawWireCylinder( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(0.5f * size);
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   S32 numPoints = sizeof(circlePoints)/sizeof(Point2F);
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoints * 4 + 4, GFXBufferTypeVolatile);
   verts.lock();
   for (S32 i=0; i<numPoints+1; i++)
   {
      S32 imod = i % numPoints;
      verts[i].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, 1.0f);
      verts[i].color = color;
      verts[i + numPoints + 1].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, -1.0f);
      verts[i + numPoints + 1].color = color;
   }

   S32 start = 2 * (numPoints+1);
   for (S32 i=0; i< numPoints; i++)
   {
      S32 idx = i & (~1); // just draw the even ones
      F32 z = i & 1 ? 1.0f : -1.0f;
      verts[start + i].point = Point3F(circlePoints[idx].x,circlePoints[idx].y, z);
      verts[start + i].color = color;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXLineStrip, 0, numPoints );
   mDevice->drawPrimitive( GFXLineStrip, numPoints + 1, numPoints);
   for (S32 i=0; i<numPoints/2; i++)
      mDevice->drawPrimitive(GFXLineStrip, 2 * numPoints + 2 + i * 2, 1);

   GFX->popWorldMatrix();
}

//-----------------------------------------------------------------------------
// Draw capsule
//-----------------------------------------------------------------------------

void GFXDrawUtil::drawSolidCapsule( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(0.5f * size);
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   S32 numPoints = sizeof(circlePoints)/sizeof(Point2F);
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoints * 2 + 2, GFXBufferTypeVolatile);
   verts.lock();
   for (S32 i=0; i<numPoints + 1; i++)
   {
      S32 imod = i % numPoints;
      verts[2 * i].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, 1.0f);
      verts[2 * i].color = color;
      verts[2 * i + 1].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, -1.0f);
      verts[2 * i + 1].color = color;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleStrip, 0, 2 * numPoints);

   GFX->popWorldMatrix();

   F32 radius = 0.5f * size.x;
   Point3F sphereCenter = pos;
   sphereCenter.z += 0.5f * size.z;
   drawSolidSphere(radius,sphereCenter,color,true,false);
   sphereCenter.z -= size.z;
   drawSolidSphere(radius,sphereCenter,color,false,true);
}

void GFXDrawUtil::drawWireCapsule( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(0.5f * size);
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   S32 numPoints = sizeof(circlePoints)/sizeof(Point2F);
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoints, GFXBufferTypeVolatile);
   verts.lock();
   for (S32 i=0; i< numPoints; i++)
   {
      S32 idx = i & (~1); // just draw the even ones
      F32 z = i & 1 ? 1.0f : -1.0f;
      verts[i].point = Point3F(circlePoints[idx].x,circlePoints[idx].y, z);
      verts[i].color = color;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   for (S32 i=0; i<numPoints; i += 2)
      mDevice->drawPrimitive(GFXLineStrip, i, 1);

   GFX->popWorldMatrix();

   F32 radius = 0.5f * size.x;
   Point3F sphereCenter = pos;
   sphereCenter.z += 0.5f * size.z;
   drawWireSphere(radius,sphereCenter,color,true,false);
   sphereCenter.z -= size.z;
   drawWireSphere(radius,sphereCenter,color,false,true);

}

//-----------------------------------------------------------------------------
// Draw cone
//-----------------------------------------------------------------------------

void GFXDrawUtil::drawSolidCone( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(0.5f * size);
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   S32 numPoints = sizeof(circlePoints)/sizeof(Point2F);
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoints + 2, GFXBufferTypeVolatile);
   verts.lock();
   verts[0].point = Point3F(0.0f,0.0f,1.0f);
   verts[0].color = color;
   for (S32 i=0; i<numPoints + 1; i++)
   {
      S32 imod = i % numPoints;
      verts[i + 1].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, -1.0f);
      verts[i + 1].color = color;
   }
   verts.unlock();
   
   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleFan, 0, numPoints );
   mDevice->drawPrimitive( GFXTriangleFan, 1, numPoints-1 );

   GFX->popWorldMatrix();
}

void GFXDrawUtil::drawWireCone( const Point3F &size, const Point3F &pos, const ColorI &color )
{
   MatrixF mat = MatrixF::Identity;
   mat.scale(0.5f * size);
   mat.setPosition(pos);
   GFX->pushWorldMatrix();
   GFX->multWorld(mat);

   S32 numPoints = sizeof(circlePoints)/sizeof(Point2F);
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, numPoints * 4 + 4, GFXBufferTypeVolatile);
   verts.lock();
   for (S32 i=0; i<numPoints+1; i++)
   {
      S32 imod = i % numPoints;
      verts[i].point = Point3F(circlePoints[imod].x,circlePoints[imod].y, -1.0f);
      verts[i].color = color;
   }

   S32 start = numPoints+1;
   for (S32 i=0; i< numPoints; i++)
   {
      S32 idx = i & (~1); // just draw the even ones
      if (i&1)
         verts[start + i].point = Point3F(0.0f,0.0f,1.0f);
      else
      verts[start + i].point = Point3F(circlePoints[idx].x,circlePoints[idx].y, -1.0f);
      verts[start + i].color = color;
   }
   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXLineStrip, 0, numPoints );
   for (S32 i=0; i<numPoints/2; i++)
      mDevice->drawPrimitive(GFXLineStrip, numPoints + 1 + i * 2, 1);

   GFX->popWorldMatrix();
}

//-----------------------------------------------------------------------------
// Draw Rectangle
//-----------------------------------------------------------------------------
void GFXDrawUtil::drawRect( const Point2I &upperLeft, const Point2I &lowerRight, const ColorI &color ) 
{
   drawRect( Point2F((F32)upperLeft.x,(F32)upperLeft.y),Point2F((F32)lowerRight.x,(F32)lowerRight.y),color);
}

void GFXDrawUtil::drawRect( const RectI &rect, const ColorI &color )
{
   drawRect( rect.point, Point2I(rect.extent.x + rect.point.x, rect.extent.y + rect.point.y), color );
}

void GFXDrawUtil::drawRect( const RectF &rect, const ColorI &color )
{
   drawRect( rect.point, Point2F(rect.extent.x + rect.point.x, rect.extent.y + rect.point.y), color );
}

void GFXDrawUtil::drawRect( const Point2F &upperLeft, const Point2F &lowerRight, const ColorI &color )
{
   //
   // Convert Box   a----------x
   //               |          |
   //               x----------b
   //
   // Into Triangle-Strip Outline
   //               v0-----------v2
   //               | a         x |
   //					  |  v1-----v3  |
   //               |   |     |   |
   //               |  v7-----v5  |
   //               | x         b |
   //               v6-----------v4
   //

   // NorthWest and NorthEast facing offset vectors
   // These adjust the thickness of the line, it'd be neat if one day
   // they were passed in as arguments.
   Point2F nw(-0.5f,-0.5f); /*  \  */
   Point2F ne(0.5f,-0.5f); /*  /  */

   GFXVertexBufferHandle<GFXVertexPC> verts (mDevice, 10, GFXBufferTypeVolatile );
   verts.lock();

   F32 ulOffset = 0.5f - mDevice->getFillConventionOffset();

   verts[0].point.set( upperLeft.x + ulOffset + nw.x, upperLeft.y + ulOffset + nw.y, 0.0f );
   verts[1].point.set( upperLeft.x + ulOffset - nw.x, upperLeft.y + ulOffset - nw.y, 0.0f );
   verts[2].point.set( lowerRight.x + ne.x, upperLeft.y + ulOffset + ne.y, 0.0f );
   verts[3].point.set( lowerRight.x - ne.x, upperLeft.y + ulOffset - ne.y, 0.0f );
   verts[4].point.set( lowerRight.x - nw.x, lowerRight.y - nw.y, 0.0f );
   verts[5].point.set( lowerRight.x + nw.x, lowerRight.y + nw.y, 0.0f );
   verts[6].point.set( upperLeft.x + ulOffset - ne.x, lowerRight.y - ne.y, 0.0f );
   verts[7].point.set( upperLeft.x + ulOffset + ne.x, lowerRight.y + ne.y, 0.0f );
   verts[8].point.set( upperLeft.x + ulOffset + nw.x, upperLeft.y + ulOffset + nw.y, 0.0f ); // same as 0
   verts[9].point.set( upperLeft.x + ulOffset - nw.x, upperLeft.y + ulOffset - nw.y, 0.0f ); // same as 1

   for (int i=0; i<10; i++)
      verts[i].color = color;

   verts.unlock();
   mDevice->setVertexBuffer( verts );

   mDevice->setStateBlock(mRectFillSB);
   mDevice->setupGenericShaders();
   mDevice->drawPrimitive( GFXTriangleStrip, 0, 8 );
}

//-----------------------------------------------------------------------------
// Draw Rectangle Fill
//-----------------------------------------------------------------------------
void GFXDrawUtil::drawRectFill( const RectF &rect, const ColorI &color )
{
   drawRectFill(rect.point, Point2F(rect.extent.x + rect.point.x, rect.extent.y + rect.point.y), color );
}

void GFXDrawUtil::drawRectFill( const Point2I &upperLeft, const Point2I &lowerRight, const ColorI &color ) 
{   
   drawRectFill(Point2F((F32)upperLeft.x, (F32)upperLeft.y), Point2F((F32)lowerRight.x, (F32)lowerRight.y), color);
}

void GFXDrawUtil::drawRectFill( const RectI &rect, const ColorI &color )
{
   drawRectFill(rect.point, Point2I(rect.extent.x + rect.point.x, rect.extent.y + rect.point.y), color );
}

void GFXDrawUtil::drawRectFill( const Point2F &upperLeft, const Point2F &lowerRight, const ColorI &color )
{
   //
   // Convert Box   a----------x
   //               |          |
   //               x----------b
   // Into Quad
   //               v0---------v1
   //               | a       x |
   //					  |           |
   //               | x       b |
   //               v2---------v3
   //

   // NorthWest and NorthEast facing offset vectors
   Point2F nw(-0.5,-0.5); /*  \  */
   Point2F ne(0.5,-0.5); /*  /  */

   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, 4, GFXBufferTypeVolatile);
   verts.lock();

   F32 ulOffset = 0.5f - mDevice->getFillConventionOffset();
   
   verts[0].point.set( upperLeft.x+nw.x+ulOffset, upperLeft.y+nw.y+ulOffset, 0.0f );
   verts[1].point.set( lowerRight.x+ne.x, upperLeft.y+ne.y+ulOffset, 0.0f );
   verts[2].point.set( upperLeft.x-ne.x+ulOffset, lowerRight.y-ne.y, 0.0f );
   verts[3].point.set( lowerRight.x-nw.x, lowerRight.y-nw.y, 0.0f );

   for (int i=0; i<4; i++)
      verts[i].color = color;

   verts.unlock();

   mDevice->setStateBlock(mRectFillSB);
   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();
   mDevice->drawPrimitive( GFXTriangleStrip, 0, 2 );
}

//-----------------------------------------------------------------------------
// Draw Line
//-----------------------------------------------------------------------------
void GFXDrawUtil::drawLine( const Point3F &startPt, const Point3F &endPt, const ColorI &color )
{
   drawLine( startPt.x, startPt.y, startPt.z, endPt.x, endPt.y, endPt.z, color );
}

void GFXDrawUtil::drawLine( const Point2F &startPt, const Point2F &endPt, const ColorI &color )
{
   drawLine( startPt.x, startPt.y, 0.0f, endPt.x, endPt.y, 0.0f, color );
}

void GFXDrawUtil::drawLine( const Point2I &startPt, const Point2I &endPt, const ColorI &color )
{
   drawLine( startPt.x, startPt.y, 0.0f, endPt.x, endPt.y, 0.0f, color );
}

void GFXDrawUtil::drawLine( F32 x1, F32 y1, F32 x2, F32 y2, const ColorI &color )
{
   drawLine( x1, y1, 0.0f, x2, y2, 0.0f, color );
}

void GFXDrawUtil::drawLine( F32 x1, F32 y1, F32 z1, F32 x2, F32 y2, F32 z2, const ColorI &color )
{
   //
   // Convert Line   a----------b
   //
   // Into Quad      v0---------v1
   //                 a         b
   //                v2---------v3
   //

   Point2F start(x1, y1);
   Point2F end(x2, y2);
   Point2F perp, lineVec;

   // handle degenerate case where point a = b
   if(x1 == x2 && y1 == y2)
   {
      perp.set(0.0f, 0.5f);
      lineVec.set(0.1f, 0.0f);
   }
   else
   {
      perp.set(start.y - end.y, end.x - start.x);
      lineVec.set(end.x - start.x, end.y - start.y);
      perp.normalize(0.5f);
      lineVec.normalize(0.1f);
   }
   start -= lineVec;
   end   += lineVec;

   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, 4, GFXBufferTypeVolatile);
   verts.lock();

   verts[0].point.set( start.x+perp.x, start.y+perp.y, z1 );
   verts[1].point.set( end.x+perp.x, end.y+perp.y, z2 );
   verts[2].point.set( start.x-perp.x, start.y-perp.y, z1 );
   verts[3].point.set( end.x-perp.x, end.y-perp.y, z2 );

   verts[0].color = color;
   verts[1].color = color;
   verts[2].color = color;
   verts[3].color = color;

   verts.unlock();
   mDevice->setVertexBuffer( verts );
   mDevice->setStateBlock(mRectFillSB);
   mDevice->drawPrimitive( GFXTriangleStrip, 0, 2 );
}

void GFXDrawUtil::drawFrustum(const Frustum& f, const ColorI &color)
{
   const Point3F  *points = f.getPoints();

   // Draw near and far planes.
   for (U32 offset = 0; offset < 8; offset+=4)
   {      
      drawLine(points[offset+0], points[offset+1], color);
      drawLine(points[offset+2], points[offset+3], color);
      drawLine(points[offset+0], points[offset+2], color);
      drawLine(points[offset+1], points[offset+3], color);            
   }

   // connect the near and far planes
   drawLine(points[Frustum::NearTopLeft], points[Frustum::FarTopLeft], color);
   drawLine(points[Frustum::NearTopRight], points[Frustum::FarTopRight], color);
   drawLine(points[Frustum::NearBottomLeft], points[Frustum::FarBottomLeft], color);
   drawLine(points[Frustum::NearBottomRight], points[Frustum::FarBottomRight], color);
}

void GFXDrawUtil::drawSolidPlane( const Point3F &pos, const Point2F &size, const ColorI &color )
{
   GFXVertexBufferHandle<GFXVertexPC> verts(mDevice, 4, GFXBufferTypeVolatile);
   verts.lock();

   verts[0].point = pos + Point3F( -size.x / 2.0f, -size.y / 2.0f, 0 );
   verts[0].color = color;
   verts[1].point = pos + Point3F( -size.x / 2.0f, size.y / 2.0f, 0 );
   verts[1].color = color;
   verts[2].point = pos + Point3F( size.x / 2.0f, size.y / 2.0f, 0 );
   verts[2].color = color;
   verts[3].point = pos + Point3F( size.x / 2.0f, -size.y / 2.0f, 0 );
   verts[3].color = color;

   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXTriangleFan, 0, 2 );
}

void GFXDrawUtil::drawPlaneGrid( const Point3F &pos, const Point2F &size, const Point2F &step, const ColorI &color )
{
   U32 xSteps = 0;
   if ( step.x > 0 )
      xSteps = size.x / step.x;

   U32 ySteps = 0;
   if ( step.y > 0 )
      ySteps = size.y / step.y;

   GFXVertexBufferHandle<GFXVertexPC> verts( mDevice, ( xSteps * 2 ) + ( ySteps * 2 ), GFXBufferTypeVolatile );
   verts.lock();

   U32 vertCount = 0;
   
   Point3F origin( pos.x - ( size.x / 2.0f ), pos.y - ( size.y / 2.0f ), pos.z );

   F32 start = mCeil( origin.x / step.x ) * step.x;

   for ( U32 i = 0; i < xSteps; i++ )
   {
      verts[vertCount].point = Point3F( start + step.x * i, origin.y, origin.z );
      verts[vertCount].color = color;
      ++vertCount;

      verts[vertCount].point = Point3F( start + step.x * i, origin.y + size.y, origin.z );
      verts[vertCount].color = color;
      ++vertCount;
   }

   start = mCeil( origin.y / step.y ) * step.y;

   for ( U32 i = 0; i < ySteps; i++ )
   {
      verts[vertCount].point = Point3F( origin.x, start + step.y * i, origin.z );
      verts[vertCount].color = color;
      ++vertCount;

      verts[vertCount].point = Point3F( origin.x + size.x, start + step.y * i, origin.z );
      verts[vertCount].color = color;
      ++vertCount;
   }

   verts.unlock();

   if ( color.alpha != 255 )
      mDevice->setStateBlock( mTrasparentSB );
   else
      mDevice->setStateBlock( mSolidSB );

   mDevice->setVertexBuffer( verts );
   mDevice->setupGenericShaders();

   mDevice->drawPrimitive( GFXLineList, 0, xSteps + ySteps );
}

