//-----------------------------------------------------------------------------
// Torque
// Copyright GarageGames, LLC 2011
// Copyright Nathan Martin 2011-2013
//-----------------------------------------------------------------------------
#include <algorithm>

// this is a small hack to prevent including in tAlgorithm.h's swap template function
// that would have broken STL's existing swap template causing compiling errors upon
// usage of std::sort() within this source file. --Nathan Martin
#define _TALGORITHM_H_

// simSet needs this from tAlgorithm.h so we will have a copy of it here.
/// Finds the first matching value within the container
/// returning the the element or last if its not found.
template <class Iterator, class Value>
Iterator find(Iterator first, Iterator last, Value value)
{
   while (first != last && *first != value)
      ++first;
   return first;
}

#include "platform/platform.h"
#include "gui/controls/guiModernTextListCtrl.h"

#include "gui/core/guiCanvas.h"
#include "console/consoleTypes.h"
#include "console/console.h"
#include "gui/containers/guiScrollCtrl.h"
#include "gui/core/guiDefaultControlRender.h"
#include "gfx/gfxDrawUtil.h"

#if !defined(SUPPORT_TGEA)
#	include "console/engineAPI.h"
#endif // !SUPPORT_TGEA


IMPLEMENT_CONOBJECT(GuiModernTextListCtrl);

#if !defined(SUPPORT_TGEA)
ConsoleDocClass( GuiModernTextListCtrl,
	"@brief GUI control that displays a tab delimited text list with a column header. Text row items in the list can be individually selected.\n\n"

	"@tsexample\n"

	"      new GuiModernTextListCtrl(GameServerList)\n"
	"		{\n"
	"			columns = \"0 256\";\n"
	"	        fitParentWidth = \"1\";\n"
	"			clipColumnText = \"0\";\n"
	"		    //Properties not specific to this control have been omitted from this example.\n"
	"		};\n"
	"@endtsexample\n\n"

	"@see Reference\n\n"

	"@ingroup GuiControls\n"
);
#endif // !SUPPORT_TGEA


//=============================================================================
// Control Callbacks
//=============================================================================

IMPLEMENT_CALLBACK( GuiModernTextListCtrl, onColumnMoved, void, (S32 colKey, U32 colOldPos, U32 colNewPos), (colKey, colOldPos, colNewPos),
	"@brief Called when the user has dragged and dropped a column header to new position.\n\n"
	"@param colKey Key associated with the column that was moved.\n"
	"@param colOldPos Column's previous index position in the header.\n"
	"@param colNewPos Column's new and current index position in the header.\n"

	"@tsexample\n"
	"function GuiModernTextListCtrl::OnColumnMoved(%this, %colKey, %colOldPos, %colNewPos)\n"
	"{\n"
	"	// User has dragged and dropped a column to another position.\n"
	"}\n"
	"@endtsexample\n\n"
	"@see GuiControl\n\n"
)
#if defined(SUPPORT_TGEA)
{
   if( isMethod("onColumnMoved") )
	   Con::executef(this, "onColumnMoved", Con::getIntArg(colKey), Con::getIntArg(colOldPos), Con::getIntArg(colNewPos));
}
#endif // SUPPORT_TGEA
;

IMPLEMENT_CALLBACK( GuiModernTextListCtrl, onColumnResized, void, (S32 colKey, U32 colPos, U32 width), (colKey, colPos, width),
	"@brief Called when the user has resized the column header and just now let go of the resizer handle.\n\n"
	"@param colKey Key associated with the column that was just resized.\n"
	"@param colPos Index position of the column within the list's header.\n"
	"@param width The new and current width of the column.\n"

	"@tsexample\n"
	"function GuiModernTextListCtrl::OnColumnResized(%this, %colKey, %colPos, %width)\n"
	"{\n"
	"	// User has resized the column in the list's header and just now let go of the resizer.\n"
	"}\n"
	"@endtsexample\n\n"
	"@see GuiControl\n\n"
)
#if defined(SUPPORT_TGEA)
{
   if( isMethod("onColumnResized") )
	   Con::executef(this, "onColumnResized", Con::getIntArg(colKey), Con::getIntArg(colPos), Con::getIntArg(width));
}
#endif // SUPPORT_TGEA
;

IMPLEMENT_CALLBACK( GuiModernTextListCtrl, onColumnSelect, void, (S32 colKey, U32 colPos, S32 sortOrder, bool doubleClick), (colKey, colPos, sortOrder, doubleClick),
	"@brief Called when the user has clicked on a column header.\n\n"
	"@param colKey Key associated with the column that was just clicked.\n"
	"@param colPos Index position of the column that was just clicked.\n"
	"@param sortOrder Current sort order of the column that was just clicked.\n"
	"@param doubleClick Value of one or true when this a double-click event.\n"

	"@tsexample\n"
	"function GuiModernTextListCtrl::OnColumnSelect(%this, %colKey, %colPos, %sortOrder, %doubleClick)\n"
	"{\n"
	"	// User has clicked on the column header and it's up to this function to decide on the response action to this event.\n"
	"	// Such as the column's sort order rule could be changed to resort the column or do nothing to not allow the the user\n"
	"	// to resort the column.\n"
	"}\n"
	"@endtsexample\n\n"
	"@see GuiControl\n\n"

)
#if defined(SUPPORT_TGEA)
{
   if( isMethod("onColumnSelect") )
	   Con::executef(this, "onColumnSelect", Con::getIntArg(colKey), Con::getIntArg(colPos), Con::getIntArg(sortOrder), Con::getIntArg(doubleClick));
}
#endif // SUPPORT_TGEA
;

IMPLEMENT_CALLBACK( GuiModernTextListCtrl, onRowSelect, void, (S32 rowKey, U32 rowPos, S32 colKey, U32 colPos, bool doubleClick), (rowKey, rowPos, colKey, colPos, doubleClick),
	"@brief Called when the user has selected a text row. Still called whether or not the selection is different since last time.\n\n"
	"@param rowKey Key associated with the new and currently selected row.\n"
	"@param rowPos Index position of the new and currently selected row.\n"
	"@param colKey Key associated with the clicked on column during row selection.\n"
	"@param colPos Index position of the clicked on column during row selection.\n"
	"@param doubleClick Value of one or true when this a double-click event.\n"

	"@tsexample\n"
	"function GuiModernTextListCtrl::OnRowSelect(%this, %rowKey, %rowPos, %colKey, %colPos, %doubleClick)\n"
	"{\n"
	"	// User has clicked on a text row item and this event still fires even though the selection may not have changed.\n"
	"}\n"
	"@endtsexample\n\n"
	"@see GuiControl\n\n"
)
#if defined(SUPPORT_TGEA)
{
   if( isMethod("onRowSelect") )
	   Con::executef(this, "onRowSelect", Con::getIntArg(rowKey), Con::getIntArg(rowPos), Con::getIntArg(colKey), Con::getIntArg(colPos), Con::getIntArg(doubleClick));
}
#endif // SUPPORT_TGEA
;

IMPLEMENT_CALLBACK( GuiModernTextListCtrl, onRowUnselect, void, (), (),
	"@brief Called when the user has unselected a text row and no other row was selected. Also called when the selected row is removed from the list.\n\n"

	"@tsexample\n"
	"function GuiModernTextListCtrl::OnRowUnselect(%this)\n"
	"{\n"
	"	// User has deselected a text row item and no other items are selected.\n"
	"}\n"
	"@endtsexample\n\n"
	"@see GuiControl\n\n"
)
#if defined(SUPPORT_TGEA)
{
   if( isMethod("onRowUnselect") )
	   Con::executef(this, "onRowUnselect");
}
#endif // SUPPORT_TGEA
;

IMPLEMENT_CALLBACK( GuiModernTextListCtrl, onDeleteKey, void, (U32 index, S32 key), (index, key),
   "@brief Called when the delete key has been pressed.\n\n"
   "@param index Index position of the row that is currently selected.\n"
   "@param key The key associated with the row that is currently selected.n"
   "@tsexample\n"
   "// The delete key was pressed while the GuiTextListCtrl was in focus, causing the callback to occur.\n"
   "function GuiModernTextListCtrl::onDeleteKey(%this, %index, %key)\n"
   "{\n"
   "	// Code to run when the delete key is pressed\n"
   "}\n"
   "@endtsexample\n\n"
   "@see GuiControl\n\n"
)
#if defined(SUPPORT_TGEA)
{
   if( isMethod("onDeleteKey") )
	   Con::executef(this, "onDeleteKey", Con::getIntArg(index), Con::getIntArg(key));
}
#endif // SUPPORT_TGEA
;


//=============================================================================
// Utility Functions
//=============================================================================

static char *getColumnContent(const char *text, char *storeBuff, U32 storeSize, S32 column)
{
	U32 len;


	// default storage buffer to empty string
	*storeBuff = 0;

	// find the requested column content, tab delimited
	while(column--)
	{
		text = dStrchr(text, '\t');
		if(!text)
			break;

		text++;
	}

	// did we find the column?
	if(column >= 0)
		return storeBuff; // nope

	// we did, now figure out the length of this column
	len = dStrcspn(text, "\t");

	// make sure we do not overflow
	if(len >= storeSize)
		len = storeSize -1;

	// copy the column content into the provided storage buffer
	dStrncpy(storeBuff, text, len);
	storeBuff[len] = 0;

	// done
	return storeBuff;
}

static S32 getHexVal(char c)
{
	if(c >= '0' && c <= '9')
		return c - '0';
	else if(c >= 'A' && c <= 'Z')
		return c - 'A' + 10;
	else if(c >= 'a' && c <= 'z')
		return c - 'a' + 10;
	return -1;
}



//=============================================================================
// GuiModernTextListCtrl Class
//=============================================================================

GuiModernTextListCtrl::GuiModernTextListCtrl()
{
//	VECTOR_SET_ASSOCIATION(mColumns);
//	VECTOR_SET_ASSOCIATION(mSortRules);
//	VECTOR_SET_ASSOCIATION(mRows);

	mActive				= true;
	mSize.set(1, 1);
	mSelectedCell		= Point2I(eGUI_InvalidIndex, eGUI_InvalidIndex);
	mCellSize			= Point2I(80, 40);	// row height 40 pixels
	mHeaderDim			= Point2I(0, 40);	// header height 40 pixels
	mHeaderProfile		= NULL;
	mSortProfile		= NULL;
	mHeaderTextPadding	= Point2I(2, 4);
	mRowTextPadding		= mHeaderTextPadding;
	mHeaderGlowOffset	= Point2I(0, 0);
	mRowGlowOffset		= mHeaderGlowOffset;
	mAllowColumnResize	= true;
	mAllowColumnMove	= true;
	mUseMarkup			= true;
	mResizeOnChange		= true;

	mThemedHeaderCell	= false;
	mThemedRowCell		= false;
	mThemedSortCell		= false;

	mColumnDND.active	= false;
	mColumnDND.column	= NULL;
	mColumnDND.move		= false;
	mColumnDND.resize	= false;
	mColumnDND.startPos	= 0;

	mDeferred.column	= NULL;
	mDeferred.row		= NULL;
	mDeferred.isDraw	= false;

	mSortManager		= new GuiModernSort(this, &mColumns, &mSortRules);

}

GuiModernTextListCtrl::~GuiModernTextListCtrl()
{
	// clean up
	delete mSortManager;

	clearSortOrders();
	clearRows();
	clearColumns();

}

void GuiModernTextListCtrl::initPersistFields()
{
	addGroup("Control");
	addProtectedField("headerProfile", TypeGuiProfile, Offset(mHeaderProfile, GuiModernTextListCtrl), &setHeaderProfileProt, &defaultProtectedGetFn,
						"The control profile that determines fill styles, font settings, etc. for column headers" );
	addProtectedField("sortProfile", TypeGuiProfile, Offset(mSortProfile, GuiModernTextListCtrl), &setSortProfileProt, &defaultProtectedGetFn,
						"The control profile that determines fill styles, font settings, etc. for sort order indicator" );

	// texture glow offset
	addField("headerGlowOffset", TypePoint2I, Offset(mHeaderGlowOffset, GuiModernTextListCtrl),
						"Number of pixels that will need to be offset when drawing a column header cell's texture. First number is for left and right and second is for top and bottom." );
	addField("rowGlowOffset", TypePoint2I, Offset(mRowGlowOffset, GuiModernTextListCtrl),
						"Number of pixels that will need to be offset when drawing a row column's texture. First number is for left and right and second is for top and bottom." );

	// text cell padding
	addField("headerTextPadding", TypePoint2I, Offset(mHeaderTextPadding, GuiModernTextListCtrl),
						"Number of pixels for padding text when drawing column header cells. First number is for horizontal and second is for vertical offsets." );
	addField("rowTextPadding", TypePoint2I, Offset(mRowTextPadding, GuiModernTextListCtrl),
						"Number of pixels for padding text when drawing row column cells. First number is for horizontal and second is for vertical offsets." );

	// preferences
	addField("allowColumnResize", TypeBool, Offset(mAllowColumnResize, GuiModernTextListCtrl),
						"Whether or not user is allowed to resize columns using the mouse cursor");
	addField("allowColumnMove", TypeBool, Offset(mAllowColumnMove, GuiModernTextListCtrl),
						"Whether or not user is allowed to move columns via drag and drop");
	addField("useMarkup", TypeBool, Offset(mUseMarkup, GuiModernTextListCtrl),
						"Whether the control should use or just ignore markup language within row contents");
	addField("resizeOnChange", TypeBool, Offset(mResizeOnChange, GuiModernTextListCtrl),
						"Whether or not to update the control size for the scroller upon content changes, ex. Add[Row|Column], etc..");

	endGroup("Control");

	Parent::initPersistFields();
}

bool GuiModernTextListCtrl::onAdd()
{
	// let parent prepare us first
	if(!Parent::onAdd())
		return false;

	// must assign the default profile
	if(!mHeaderProfile)
	{
		GuiControlProfile *profile = NULL;
		Sim::findObject("GuiDefaultProfile", profile);

		AssertISV(profile != NULL, avar("GuiModernTextListCtrl::onAdd() unable to find specified profile and GuiDefaultProfile does not exist!"));

		setHeaderProfile(profile);
	}

	// must assign the default profile
	if(!mSortProfile)
	{
		GuiControlProfile *profile = NULL;
		Sim::findObject("GuiDefaultProfile", profile);

		AssertISV(profile != NULL, avar("GuiModernTextListCtrl::onAdd() unable to find specified profile and GuiDefaultProfile does not exist!"));

		setSortProfile(profile);
	}

	// Success.
	return true;
}

void GuiModernTextListCtrl::onRemove()
{
	// clean up
	clearColumns();
	clearRows();
	clearSortOrders();

	if(mHeaderProfile)
	{
		mHeaderProfile->decRefCount();
		mHeaderProfile = NULL;
	}
	if(mSortProfile)
	{
		mSortProfile->decRefCount();
		mSortProfile = NULL;
	}

	Parent::onRemove();
}

bool GuiModernTextListCtrl::onWake()
{
	if(!Parent::onWake())
		return false;

	// increment the profile
	mHeaderProfile->incRefCount();
	mSortProfile->incRefCount();

	mThemedRowCell		= false;
	mThemedHeaderCell	= false;

	if(mProfile->mUseBitmapArray)			mThemedRowCell		= (mProfile->constructBitmapArray()			>= 36);
	if(mHeaderProfile->mUseBitmapArray)		mThemedHeaderCell	= (mHeaderProfile->constructBitmapArray()	>= 36);

	setSize(mSize);
	return true;
}

void GuiModernTextListCtrl::onSleep()
{
	Parent::onSleep();

	// decrement the profile reference
	mHeaderProfile->decRefCount();
	mSortProfile->decRefCount();

}

bool GuiModernTextListCtrl::onKeyDown( const GuiEvent &event )
{
	//if this control is a dead end, make sure the event stops here
	if (!mVisible || !mActive || !mAwake)
		return true;

	S32 yDelta = 0;
	switch(event.keyCode)
	{
		case KEY_RETURN:
			execAltConsoleCallback();
			break;

		case KEY_LEFT:
		case KEY_UP:
			if(mSelectedCell.y > 0)
			{
				mSelectedCell.y--;
				yDelta = -mCellSize.y;
			}
			break;

		case KEY_DOWN:
		case KEY_RIGHT:
			if(mSelectedCell.y < (mRows.size() - 1))
			{
				mSelectedCell.y++;
				yDelta = mCellSize.y;
			}
			break;

		case KEY_HOME:
			if(mRows.size())
			{
				mSelectedCell.y = 0;
				yDelta = 0-(mCellSize.y * mRows.size() + 1);
			}
			break;

		case KEY_END:
			if(mRows.size())
			{
				mSelectedCell.y = mRows.size() - 1;
				yDelta = (mCellSize.y * mRows.size() + 1);
			}
			break;

		case KEY_DELETE:
			if(mSelectedCell.y >= 0 && mSelectedCell.y < mRows.size())
				onDeleteKey_callback((U32)mSelectedCell.y, mSortedRows[mSelectedCell.y]->key);
			break;

		default:
			return Parent::onKeyDown(event);
			break;
	};

	GuiScrollCtrl* parent = dynamic_cast<GuiScrollCtrl *>(getParent());
	if(parent)
		parent->scrollDelta( 0, yDelta );

	return true;
}

bool GuiModernTextListCtrl::cellSelected(Point2I cell)
{
	bool isOK = false;

	// Is the selection being cleared?
	if(cell.x == -1 && cell.y == -1)
		isOK = true;

	// Do not allow selection of inactive cells
	if(cell.y >= 0 && cell.y < mSize.y && (mSortedRows[cell.y]->flags & Row_Flags_Selectable))
		isOK = true;

	if(!isOK)
		return false;

	// set cell selection to none upon invalid cell range
	if(/*cell.x < 0 || cell.x >= mSize.x ||*/ cell.y < 0 || cell.y >= mSize.y)
	{
		mSelectedCell = Point2I(-1,-1);
		onCellSelected(mSelectedCell);
		return false;
	}

	// update current cell selection
	mSelectedCell = cell;
	scrollCellVisible(mSelectedCell);
	onCellSelected(cell);
	mSelectedCell.x = 0;
	setUpdate();

	return true;
}

void GuiModernTextListCtrl::onCellSelected(Point2I cell)
{
/*
	onSelect_callback(Con::getIntArg(mList[cell.y].id), mList[cell.y].text);
	execConsoleCallback();
*/
	tRowItem *row;
	tColItem *col;

	if(cell.y >= 0 && getRowByIndex(cell.y, row) && getColumnByIndex(cell.x, col))
	{
		// callback handler
		onRowSelect_callback(row->key, cell.y, col->key, cell.x, (mGuiEvent->mouseClickCount > 1));
	} else
	{
		// callback handler
		onRowUnselect_callback();
	}
}

Point2I GuiModernTextListCtrl::prepTextAlignment(GuiControlProfile *profile, RectI &rectCell, char *text, S32 align, Point2I &padding, S32 additional)
{
	Point2I textPos(0, 0), textSize;


	// prepare to draw text
	textSize.x	= profile->mFont->getStrWidth((const UTF8 *)text) + additional;
	textSize.y	= profile->mFont->getHeight();

	// set the text to center vertical alignment
	if(textSize.y < rectCell.extent.y)
		textPos.y = (rectCell.extent.y / 2) - (textSize.y / 2);

	// set the text horizontal alignment, default to left align
	textPos.x = padding.x;

	switch (align)
	{
		case Column_Align_Right:
		{
			textPos.x = rectCell.extent.x - textSize.x - padding.x;
			break;
		}
		case Column_Align_Center:
		{
			if(textSize.x < (rectCell.extent.x - padding.x * 2))
				textPos.x = (rectCell.extent.x / 2) - (textSize.x / 2);
			break;
		}
	}

	return textPos;
}

void GuiModernTextListCtrl::prepCellDraw(GuiControlProfile *profile, RectI &rectCell, Point2I &glowOffset, bool active, bool selected, bool mouseOver, bool texture)
{
	ColorI		cFill, cBorder, cFont;
	S32			texIndex;


	if(!active)
	{
		// cell is not active / disabled
		cFill		= profile->mFillColorNA;
		cBorder		= profile->mBorderColorNA;
		cFont		= profile->mFontColorNA;
		texIndex	= 4;

		// workaround for row selection highlight to work as expected even on inactive rows
		if(selected || (profile->mMouseOverSelected && mouseOver))
			cFill	= profile->mFillColorSEL;

	} else if(selected || (profile->mMouseOverSelected && mouseOver))
	{
		// cell is selected
		cFill		= profile->mFillColorSEL;
		cBorder		= profile->mBorderColorHL;
		cFont		= profile->mFontColorSEL;
		texIndex	= 2;

	} else if(mouseOver)
	{
		// cell is highlighted
		cFill		= profile->mFillColorHL;
		cBorder		= profile->mBorderColorHL;
		cFont		= profile->mFontColorHL;
		texIndex	= 3;

	} else
	{
		// cell is in a normal state
		cFill		= profile->mFillColor;
		cBorder		= profile->mBorderColor;
		cFont		= profile->mFontColor;
		texIndex	= 1;
	}

	// draw fill border when:
	// A) Border is requested.
	// B) Cell is selected or mouse is over the cell. 
	if(profile->mBorder || (selected || mouseOver))
	{
		// do not draw a border if not requested
		if(!profile->mBorder)
			cBorder = ColorI(0, 0, 0, 0);

		if(texture)
		{
			RectI rectTex(rectCell.point.x - glowOffset.x, rectCell.point.y - glowOffset.y, rectCell.extent.x + glowOffset.x * 2, rectCell.extent.y + glowOffset.y * 2);
			renderSizableBitmapBordersFilled(rectTex, texIndex, profile);
		}
		else
		{
			renderFilledBorder(rectCell, cBorder, cFill, profile->mBorderThickness);
		}
	}

	// set text color
	GFX->getDrawUtil()->setBitmapModulation(cFont);

}

S32 GuiModernTextListCtrl::stripMarkupText(char *dstText, const char *srcText, bool keepPrimitives, bool preservDest)
{
	const char *p, *e, *p2;
	U32 len, len2;
	S32 bitmapSize = 0;

	if(!preservDest)
		*dstText = 0;

	while(srcText)
	{
		// find opening tag
		p = dStrchr(srcText, '<');

		// did we find it?
		if(!p)
			break; // no, abort out

		// yes, copy any strings up to this point to the destination buffer
		len = p - srcText;
		if(len)
		{
			dStrncat(dstText, srcText, len);
			srcText += len;
		}

		// check to see if an escape char is next to the opening tag
		if(p[1] == '<')
		{
			// indeed, ignore opening tag
			dStrncat(dstText, srcText, 1);
			srcText += 2;
			continue;
		}
		
		// attempt to find the closing tag
		e = dStrchr(p +1, '>');
		if(!e)
		{
			// failed, the rest of the text is to be tossed
			srcText += dStrlen(srcText);
			break;
		}
		len = e - p;


		// are we to bother keeping primitive values from bitmap tags?
		// and if so specifically check for the primitive flag character.
		if(p[1] == '!')
		{
			// find the primitive terminating character
			p2 = dStrchr(p, ':');

			// make sure that we found it and that it resides within this tag's boundaries
			if(p2 && p2 < e)
			{
				len2 = (p2 - (p+2));
				if(keepPrimitives)
					dStrncat(dstText, &p[2], len2);

				// skip it
				len2	= (p2 - p);
				srcText	+= len2;
				p		+= len2;
				len		-= len2;
			}
		}

		if(!keepPrimitives)
		{
			static char		texName[512];
			tBitmap			*bitmap;

			// calculate the total width size of bitmaps used in this cell
			if(!dStrnicmp("bitmap:", &p[1], 7))
			{
				// get bitmap filename
				p += 8; len2 = len - 8;
				dStrncpy(texName, p, len2);
				texName[len2] = 0;

				// lookup the filename
				if(!getBitmap(texName, bitmap))
					goto SkipTag; // bitmap failure

				// update total bitmap width
				bitmapSize += bitmap->extent.x;
			}
		}
SkipTag:

		// move past the tag
		srcText += len +1;
	}

	// copy the remaining string into destination buffer
	dStrcat(dstText, srcText);

	// done
	return bitmapSize;
}

void GuiModernTextListCtrl::stripMarkupTextFields(char *&dstText, const char *srcText, bool keepPrimitives)
{
	static char textCol[MAXTEXTLEN], textStrip[MAXTEXTLEN];
	U32 count, i;

	*textStrip = 0;
	count = mColumns.size();
	for(i=0; i<count; i++)
	{
		// strip the column content of markup
		getColumnContent(srcText, textCol, sizeof(textCol), i);
		stripMarkupText(textStrip, textCol, keepPrimitives, true);

		// tab delimit the field
		if((i+1)<count)
			dStrcat(textStrip, "\t");
	}

	dstText = textStrip;
}

void GuiModernTextListCtrl::drawMarkupText(GuiControlProfile *profile, RectI &rectCell, Point2I &textPos, const char *text)
{
	/****

	GuiModernTextListCtrl has a limited support for torque markup in order to be as simple as possible
	and constrained to single line text cells. This is the list of supported markup tags:


	<color[:RRGGBB[AA]]>				-- Foreground text color tag

	RRGGBB[AA]			Optional tag field that declares the foreground text color when specified, else the foreground text
						color is reset back using the color specified in the control's datablock based on the row condition.

						Additionally a sub-optional field is available to specify the color's alpha value.
	

	<[!pv:]bitmap:imageFilePath>		-- Bitmap image tag

	imageFilePath		Required field that specifies the bitmap to be drawn.

	[!pv:]				Optional tag field is a unique attribute to GuiModernTextListCtrl that declares
						a primitive value for bitmap tags to specify the textual value when the markup tags
						are stripped for cases like row column sort order processing.

						Example: "<!V12:bitmap:core/art/images/logo.png>" will be replaced with "V12" during column sorting.

	****/

	enum
	{
		MLTag_Color = 1,
		MLTag_Bitmap,

		MLTag__End
	};
	struct tColumnFlags
	{
		S32			id;
		const char	*name;
		U32			len;
	};
	static tColumnFlags tags[] =
	{
		{ MLTag_Color,		"color" },
		{ MLTag_Bitmap,		"bitmap" }
	};

	// make sure tags array is initialized
	if(!tags->len)
	{
		for(U32 i=0; i<(MLTag__End -1); i++)
			tags[i].len = dStrlen(tags[i].name);
	}


	const U32		maxStackPos = 12;
	const char		*pStart, *pEnd, *p, *e;
	U32				len, lenName, stackPos = 0, i, pos;
	ColorI			colorStack[maxStackPos], colorOrig, color;
	static char		texName[512];
	tBitmap			*bitmap;


	// remember original font color
	GFX->getDrawUtil()->getBitmapModulation(&colorOrig);
	color = colorOrig;

	// prepare pointers
	pStart = pEnd = text;

	while(text)
	{
		// find opening tag
		p = dStrchr(text, '<');

		// did we find it?
		if(!p)
		{
			// no, abort out
			pEnd = text + dStrlen(text);
			break;
		}
		pEnd = p;

		// check to see if an escape char is next to the opening tag
		if(p[1] == '<')
		{
			// indeed, ignore opening tag
			pEnd	= p +1;
			text	= pEnd +1;
		}

		// go ahead and draw any pending text
		len = pEnd - pStart;
		if(len)
		{
			textPos.x	= GFX->getDrawUtil()->drawTextN(profile->mFont, textPos, pStart, len, profile->mFontColors);
			pStart		= text;
		}

		if(p[1] == '<')
			continue;

		// attempt to find the closing tag
		e = dStrchr(p +1, '>');
		if(!e)
		{
			// failed, the rest of the text is to be tossed
			text += dStrlen(text);
			pEnd = pStart;
			break;
		}

		// test for primitive value flag character and skip the field if found
		if(p[1] == '!')
		{
			// find the primitive terminating character
			p = dStrchr(p, ':');

			// upon invalid pointer position just skip the tag
			if(!(p && p < e))
				goto SkipTag;
		}

		// bump and set tag length
		p++;
		len = e - p;

		// get tag name length 
		lenName = dStrcspn(p, ":>");
		

		// find match in registered markup language tags
		for(i=0, pos=0; i<(MLTag__End -1); i++)
		{
			if(lenName != tags[i].len || dStrnicmp(p, tags[i].name, lenName))
				continue;

			// match found
			pos = tags[i].id;
			break;
		}

		// push pointer forward
		p	+= lenName;
		len	-= lenName;

		// process tag
		switch (pos)
		{
			// <color[:RRGGBB[AA]]>
			case MLTag_Color:
			{
				// are we pushing or popping the color stack?
				if(*p == '>')
				{
					// popping
					if(stackPos)
						color = colorStack[--stackPos];

					// set font color
					GFX->getDrawUtil()->setBitmapModulation(color);
					break;
				}

				// pushing
				if(stackPos < maxStackPos)
					colorStack[stackPos++] = color;

				// make sure we have the characters for color field
				if(len < 7)
					break;

				// parse the colors
				color.red	= getHexVal(p[1]) * 16 + getHexVal(p[2]);
				color.green	= getHexVal(p[3]) * 16 + getHexVal(p[4]);
				color.blue	= getHexVal(p[5]) * 16 + getHexVal(p[6]);

				if(len == 9)
					color.alpha = getHexVal(p[7]) * 16 + getHexVal(p[8]);
				else
					color.alpha = 255;

					// set font color
					GFX->getDrawUtil()->setBitmapModulation(color);
				break;
			}

			case MLTag_Bitmap:
			{
				if(*p != ':')
					break;

				// get bitmap filename
				p++; len--;
				dStrncpy(texName, p, len);
				texName[len] = 0;

				// lookup the filename
				if(!getBitmap(texName, bitmap))
					break; // bitmap failure

				// clear color modulation
				GFX->getDrawUtil()->clearBitmapModulation();

				Point2I bmPos = Point2I(textPos.x, rectCell.point.y);
				bmPos.y += (rectCell.extent.y / 2) - (bitmap->extent.y / 2);

				// draw bitmap
				GFX->getDrawUtil()->drawBitmap(bitmap->texture, bmPos);
				textPos.x += bitmap->extent.x;

				// restore color modulation
				GFX->getDrawUtil()->setBitmapModulation(color);
				break;
			}
		}


SkipTag:
		// move past the tag
		text	= e +1;
		pStart	= text;
	}

	// draw the remaining text
	len = pEnd - pStart;
	GFX->getDrawUtil()->drawTextN(profile->mFont, textPos, pStart, len, profile->mFontColors);

	// done
}

void GuiModernTextListCtrl::drawColumnSortArrow(tColItem *col, RectI &drawRect, RectI &rectText)
{
	ColorI		colorOrig;
	Point2I		pos, imgSize;


	// are we drawing a texture arrow?
	if(!mSortProfile->mTextureObject)
		return; // abort, we don't suppport manual drawing of arrows at this time

	// get arrow image dimenitions
	imgSize.x	= mSortProfile->mTextureObject.getWidth();
	imgSize.y	= mSortProfile->mTextureObject.getHeight();

	// set the center vertical alignment
	if(imgSize.y < drawRect.extent.y)
		pos.y = (drawRect.extent.y / 2) - (imgSize.y / 2);

	// set the right horizontal alignment
	pos.x = drawRect.extent.x - imgSize.x - mHeaderTextPadding.x;

	// remember currently set color
	GFX->getDrawUtil()->getBitmapModulation(&colorOrig);

	// set color associated with sort order position or priority
	GFX->getDrawUtil()->setBitmapModulation(mSortProfile->mFontColors[ col->sortPos % 10]);

	// draw bitmap
	pos += drawRect.point;
	GFX->getDrawUtil()->drawBitmap(mSortProfile->mTextureObject, pos, ((col->sort == 1)? GFXBitmapFlip_Y:GFXBitmapFlip_None));

	// restore color
	GFX->getDrawUtil()->setBitmapModulation(colorOrig);

	// try keep text off the arrow bitmap if requested
	if(mSortProfile->mBorder)
	{
		rectText.extent.x -= imgSize.x + (mSortProfile->mBorderThickness);
		if(rectText.extent.x < 1)
			rectText.extent.x = 1;
	}

}


void GuiModernTextListCtrl::onRenderColumnHeaders(Point2I offset, Point2I parentOffset, Point2I headerDim)
{
	GuiControlProfile	*profile = mHeaderProfile;
	tviColumns			it;
	Point2I				drawPos(0, 0), textPos(0, 0);
	RectI				rectClipSave, rectCell, rectClip, rectText;
	bool				active, selected, mouseOver; 


	// save current clip rectangle
	rectClipSave = GFX->getClipRect();

	// ignore vertical offset
	offset.y = parentOffset.y;

	// iterate through all columns
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		// skip invisible columns,
		// skip deferred draw columns when not drawing deferred columns,
		// skip non-deferred draw columns when drawing deferred columns
		if(!(it->flags & Column_Flags_Visible) ||
			((it->flags & Column_Flags_Deferred) && !mDeferred.isDraw) ||
			(!(it->flags & Column_Flags_Deferred) && mDeferred.isDraw))
			goto MoveOnToNext;

		// set cell drawing region
		rectCell = RectI(drawPos.x + offset.x, offset.y, it->width, headerDim.y);
		rectClip = rectCell;
		rectText = rectCell;

		// determine if its worth drawing the column
		if(!rectClip.intersect(rectClipSave))
			goto MoveOnToNext; // don't fear the goto

		// loosen the clip region upon deferred draw for glow offsets
		if(it->flags & Column_Flags_Deferred)
		{
			rectClip.point	-= mHeaderGlowOffset;
			rectClip.extent	+= mHeaderGlowOffset * 2;
		}

		// set cell clip region
		GFX->setClipRect(rectClip);

		// draw column sort indicator when bottom justify
		if(it->sort && mSortProfile->mAlignment == GuiControlProfile::BottomJustify)
			drawColumnSortArrow(&it[0], rectCell, rectText);

		// draw background
		active		= !!(it->flags & Column_Flags_Active);
		selected	= !!(it->flags & Column_Flags_Select);
		mouseOver	= !!(it->flags & Column_Flags_Highlight);
		prepCellDraw(profile, rectCell, mHeaderGlowOffset, active, selected, mouseOver, mThemedHeaderCell);

		// now re-tighten the clip region for the rest of the column draw operations
		if(it->flags & Column_Flags_Deferred)
		{
			rectClip.point	+= mHeaderGlowOffset;
			rectClip.extent	-= mHeaderGlowOffset * 2;

			// set cell clip region
			GFX->setClipRect(rectClip);
		}

		// draw sort arrow (if center justify)
		if(it->sort && mSortProfile->mAlignment == GuiControlProfile::CenterJustify)
			drawColumnSortArrow(&it[0], rectCell, rectText);

		// get text alignment position
		textPos = prepTextAlignment(profile, rectText, it->name, profile->mAlignment, mHeaderTextPadding);

		// push text start point into drawing rectangle and then draw the text
		textPos += drawPos + offset;
		GFX->getDrawUtil()->drawText(profile->mFont, textPos, it->name, profile->mFontColors);

		// draw column sort indicator when top justify
		if(it->sort && mSortProfile->mAlignment == GuiControlProfile::TopJustify)
			drawColumnSortArrow(&it[0], rectCell, rectText);


MoveOnToNext:
		
		// only update draw position on visible columns
		if(it->flags & Column_Flags_Visible)
			drawPos.x += it->width;
		
		// when deferred drawing and we've reached deferred column then stop
		if(mDeferred.isDraw && (it->flags & Column_Flags_Deferred))
			break;

		// next column
		it++;
	}

	// restore clip rectangle
//	GFX->setClipRect(rectClipSave);

	// done
}

void GuiModernTextListCtrl::onRenderCell(Point2I offset, Point2I cell, bool selected, bool mouseOver)
{
	GuiControlProfile	*profile = mProfile;
	tviColumns			it;
	tRowItem			*row;
	char				*colText;
	Point2I				drawPos(0, 0), textPos(0, 0);
	RectI				rectClipSave, rectCell, rectClip;
	static char			storeText[MAXTEXTLEN], cleanText[MAXTEXTLEN];
	S32					bitmapSize = 0;
	bool				active;


	// save current clip rectangle
	rectClipSave = GFX->getClipRect();

	// get row and abort on retrieve error
	if(!getRowByIndex(cell.y, row))
		return;

	// iterate through all columns
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		// skip invisible columns
		if(!(it->flags & Column_Flags_Visible))
			goto MoveOnToNext;

		// set cell drawing region
		rectCell = RectI(drawPos.x + offset.x, offset.y, it->width, mCellSize.y);
		rectClip = rectCell;

		// determine if its worth drawing the column
		if(!rectClip.intersect(rectClipSave))
			goto MoveOnToNext; // don't fear the goto

		// set cell clip region
		GFX->setClipRect(rectClip);

		// get column text content
		colText = getColumnContent(row->text, storeText, sizeof(storeText), it->column);

		// get a cleaned of markup text copy
		if(mUseMarkup)
		{
			bitmapSize	= stripMarkupText(cleanText, storeText, false);
			colText		= cleanText;
		}

		// get text alignment position
		textPos = prepTextAlignment(profile, rectCell, colText, it->align, mRowTextPadding, bitmapSize);

		// push text start point into drawing rectangle
		textPos += drawPos + offset;

		// draw background and then text
		active = !!(row->flags & Row_Flags_Active);
		prepCellDraw(profile, rectCell, mRowGlowOffset, active, selected, mouseOver, mThemedRowCell);
		if(mUseMarkup)
			drawMarkupText(profile, rectCell, textPos, storeText);
		else
			GFX->getDrawUtil()->drawTextN(profile->mFont, textPos, colText, dStrlen(colText), profile->mFontColors);

MoveOnToNext:
		// next column
		if(it->flags & Column_Flags_Visible)
			drawPos.x += it->width;
		it++;
	}

	// restore clip rectangle
//	GFX->setClipRect(rectClipSave);

	// done
}

void GuiModernTextListCtrl::onRender(Point2I offset, const RectI &updateRect)
{
	// save us a copy of the arguments as parent changes them
	Point2I		off(offset);
	RectI		rectUpdate(updateRect);

	// have parent draw first
	Parent::onRender(offset, updateRect);

	//make sure we have a parent
	GuiControl *parent = getParent();
	if(!parent)
		return;

	Point2I parentOffset = parent->localToGlobalCoord(Point2I(0, 0));

	// now perform deferred drawing if necessary
	if(mDeferred.column)
	{
		mDeferred.isDraw = true;
		GFX->setClipRect(updateRect);

		offset += mDeferred.pos;
		onRenderColumnHeaders(offset, parentOffset, mHeaderDim);

		mDeferred.isDraw = false;
	}
	if(mDeferred.row)
	{
	}
}


void GuiModernTextListCtrl::scrollCellVisible(Point2I cell)
{
	//make sure we have a parent
	//make sure we have a valid cell selected
	GuiScrollCtrl *parent = dynamic_cast<GuiScrollCtrl*>(getParent());
	if(!parent || cell.x < 0 || cell.y < 0)
		return;

	// grab current scroll position
	Point2I	scroll	= parent->getChildRelPos();
	RectI	cellBounds(scroll.x, cell.y * mCellSize.y, 5, mCellSize.y);

	// prevent underscrolling downward when cell is below scroll position
	if(scroll.y < cellBounds.point.y)
		cellBounds.point.y += mHeaderDim.y;

	// scroll down to requested cell to make it visible
	parent->scrollRectVisible(cellBounds);
}

void GuiModernTextListCtrl::onMouseDown(const GuiEvent &event)
{
	if(!mActive || !mAwake || !mVisible)
		return;

	if(!mColumnDND.active && mColumnDND.column != NULL && mColumnDND.resize)
	{
		mColumnDND.active = true;
		mouseLock();
	}

	if(mDeferred.column && !mColumnDND.resize && !(mDeferred.column->flags & Column_Flags_Select))
		mDeferred.column->flags |= Column_Flags_Select;

}

void GuiModernTextListCtrl::onMouseUp(const GuiEvent &event)
{
	Point2I		scroll(0, 0);
	RectI		rectHeaders;
	Point2I		pt;
	tColItem	*col;
	U32			index;


	if(!mActive || !mAwake || !mVisible)
		return;

	// let the guiControl method take care of the rest
	GuiControl::onMouseUp(event);
	mGuiEvent = &event;



	GuiScrollCtrl *parent = dynamic_cast<GuiScrollCtrl*>(getParent());
	if(parent)
	{
		// grab current scroll position
		scroll = parent->getChildRelPos();
	}

	// get hit point from mouse cursor
	pt = globalToLocalCoord(event.mousePoint);

	// find out which column the mouse was over
	hitTestColumn(pt, col, index);


	// handle deselect of deferred draw column
	if(mDeferred.column && (mDeferred.column->flags & Column_Flags_Select))
		mDeferred.column->flags &= ~Column_Flags_Select;

	// handle drag and drop events
	if(mColumnDND.active)
	{
		U32 oindex = getColumnIndexByPointer(mColumnDND.column);

		if(mColumnDND.resize)
		{
			// callback handler
			onColumnResized_callback(mColumnDND.column->key, oindex, mColumnDND.column->width);

			mColumnDND.resize = false;
		}
		if(mColumnDND.move)
		{
			// drop the deferred drawing of column
			mDeferred.column->flags &= ~Column_Flags_ClearSoft;
			mDeferred.column = NULL;

			// set the rectangle to cover the column headers, this
			// test will make it required that the user must let go
			// of the moving column while the mouse cursor is literally
			// on top of the target column in order to move the column there.
			rectHeaders = RectI(0, scroll.y, mCellSize.x, mHeaderDim.y);

			// figure out where the column was dropped to based on the mouse cursor position
			if(col != NULL && col != mColumnDND.column && rectHeaders.pointInRect(pt))
			{
				moveColumn(mColumnDND.column->key, index);

				// callback handler
				onColumnMoved_callback(mColumnDND.column->key, oindex, index);
			}

			mColumnDND.move = false;
		}

		mColumnDND.active = false;
		mouseUnlock();
		return;
	}

	// handle the column click event
	if(col != NULL)
	{
		// set the rectangle to cover the column headers
		rectHeaders = RectI(0, scroll.y, mCellSize.x, mHeaderDim.y);

		// now determine if the click event occured on a column header
		if(rectHeaders.pointInRect(pt))
		{
			tSortRule *rule;
			S32 order = -1;

			// get sort order, if any
			if(getSortRuleByKey(col->key, rule))
				order = rule->order;

			// callback handler
			onColumnSelect_callback(col->key, index, order, (event.mouseClickCount > 1));
			return;
		}
	}


	// standard array control stuff...
	pt.x -= mHeaderDim.x; pt.y -= mHeaderDim.y;
	Point2I cell(
		//(pt.x < 0 ? -1 : pt.x / mCellSize.x),
		(S32)index,
		(pt.y < 0 ? -1 : pt.y / mCellSize.y)
	);

	if(/*cell.x >= 0 && cell.x < mSize.x &&*/ cell.y >= 0 && cell.y < mSize.y)
	{
		//store the previously selected cell
		Point2I prevSelected = mSelectedCell;

		//select the new cell
		cellSelected(Point2I(cell.x, cell.y));

		//if we double clicked on the *same* cell, evaluate the altConsole Command
		if((event.mouseClickCount > 1) && (prevSelected == mSelectedCell))
			execAltConsoleCallback();
	}
}

void GuiModernTextListCtrl::onMouseDragged(const GuiEvent &event)
{
	// for the array control, the behavior of onMouseDragged is the same
	// as on mouse moved - basically just recalc the current mouse over cell
	// and set the update regions if necessary
//	GuiArrayCtrl::onMouseMove(event);


	if(!mColumnDND.active && mColumnDND.column != NULL && mColumnDND.move)
	{
		mColumnDND.active = true;
		mouseLock();
	}

	if(!mColumnDND.active)
		return;

	Point2I pt = globalToLocalCoord(event.mousePoint);

	if(mColumnDND.resize)
	{
		tColItem	*col	= mColumnDND.column;
		S32			width;

		// now resize the column header based on provided mouse cursor vertical position
		width = pt.x - mColumnDND.startPos;
		if(width < 0)
			width = 0;

		// enforce width limits
		if(width < col->minWidth)	width = col->minWidth;
		if(width > col->maxWidth)	width = col->maxWidth;

		// now set column width and update our control size
		col->width = width;
		updateSize();
	}
	if(mColumnDND.move)
	{
		if(mDeferred.column == NULL)
		{
			mDeferred.column = mColumnDND.column;
			mDeferred.column->flags |= Column_Flags_Deferred;
		}

		// update dragged location of column
		mDeferred.pos.y = 0;
		mDeferred.pos.x = pt.x - mColumnDND.startPos;
	}

}

void GuiModernTextListCtrl::onMouseLeave(const GuiEvent &event)
{
	// notify parent
	Parent::onMouseLeave(event);

	// abort when column drag and drop are active
	if(mColumnDND.active || !mDeferred.column)
		return;

	// clear any deferred drawing
	mDeferred.column->flags &= ~Column_Flags_ClearSoft;
	mDeferred.column = NULL;
}

void GuiModernTextListCtrl::checkColumnResizers(const GuiEvent &lastGuiEvent)
{
	tviColumns		it;
	RectI			rectColumn;
	Point2I			scroll(0, 0), pt;
	S32				start, pos = 0;
	bool			result = false;


	// don't bother if user is currently resizing or moving a column header
	if(mColumnDND.active && (mColumnDND.resize || mColumnDND.move))
		return;

	GuiScrollCtrl *parent = dynamic_cast<GuiScrollCtrl*>(getParent());
	if(parent)
	{
		// grab current scroll position
		scroll = parent->getChildRelPos();
	}

	// iterate through all column headers and check to see if the mouse cursor
	// is within a 4 pixel range of the right-side of the column header.
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		// skip invisible columns
		if(!(it->flags & Column_Flags_Visible))
			goto MoveOnToNext;

		pt		 = globalToLocalCoord(lastGuiEvent.mousePoint);
		start	 = pos;
		pos		+= it->width;

		// set column resizer rectangle to within 2 pixel range of its right-hand side border
		rectColumn = RectI(pos - 2, scroll.y, 4, mHeaderDim.y);

		// is cursor within this column's resizer rectangle?
		if(mAllowColumnResize && rectColumn.pointInRect(pt))
		{
			// yes
			result				= true;
			mColumnDND.resize	= true;
			mColumnDND.move		= false;
			mColumnDND.column	= &it[0];
			mColumnDND.startPos	= start;

			// clear deferred column
			if(mDeferred.column)
			{
				mDeferred.column->flags &= ~Column_Flags_ClearSoft;
				mDeferred.column = NULL;
			}
			break;
		}

		// set column rectangle to match the column's dimensions
		rectColumn = RectI(start, scroll.y, it->width, mHeaderDim.y);

		// is cursor within this column's rectangle?
		if(rectColumn.pointInRect(pt))
		{
			if(mAllowColumnMove)
			{
				// yes
				result				= true;
				mColumnDND.move		= true;
				mColumnDND.resize	= false;
				mColumnDND.column	= &it[0];
				mColumnDND.startPos	= start + (pt.x - start);
			}

			// setup mouse hot column for deferred drawing
			if(mDeferred.column != &it[0])
			{
				// unhook existing deferred column
				if(mDeferred.column)
					mDeferred.column->flags &= ~Column_Flags_ClearSoft;

				// set this column as deferred and highlight this column
				mDeferred.column	= &it[0];
				mDeferred.pos.x		= 0;
				mDeferred.pos.y		= 0;
				mDeferred.column->flags |= Column_Flags_Deferred | Column_Flags_Highlight;
			}
			break;
		}

MoveOnToNext:
		// next column
		it++;
	}

	// done
	if(!result)
	{
//		mColumnDND.active	= false;
		mColumnDND.move		= false;
		mColumnDND.resize	= false;
		mColumnDND.column	= NULL;

		// clear deferred draw column
		if(mDeferred.column)
		{
			mDeferred.column->flags &= ~Column_Flags_ClearSoft;
			mDeferred.column = NULL;
		}
	}
}

void GuiModernTextListCtrl::getCursor(GuiCursor *&cursor, bool &showCursor, const GuiEvent &lastGuiEvent)
{
	GuiCanvas *rootCtrl = getRoot();
	if(!rootCtrl)
		return;

	PlatformWindow *platformWindow = static_cast<GuiCanvas*>(getRoot())->getPlatformWindow();
	AssertFatal( platformWindow != NULL,"GuiControl without owning platform window!  This should not be possible." );

	PlatformCursorController *cusrorController = platformWindow->getCursorController();
	AssertFatal( cusrorController != NULL,"PlatformWindow without an owned CursorController!" );

	// check to see if mouse cursor is near any column header resizers
	checkColumnResizers(lastGuiEvent);

	// Check to see if we need one or just the default...
	if(mColumnDND.resize || (mColumnDND.active && mColumnDND.move))
	{
		S32 desiredCursor = (mColumnDND.resize)? PlatformCursorController::curResizeVert : PlatformCursorController::curResizeAll;

		// Do we need to change it or is it already set?
		if(rootCtrl->mCursorChanged != desiredCursor)
		{
			// We've already changed the cursor, so set it back
			if(rootCtrl->mCursorChanged != -1)
				cusrorController->popCursor();

			// Now change the cursor shape
			cusrorController->pushCursor(desiredCursor);
			rootCtrl->mCursorChanged = desiredCursor;
		}
	}
	else if(rootCtrl->mCursorChanged != -1)
	{
		// Just the default
		cusrorController->popCursor();
		rootCtrl->mCursorChanged = -1;
	}
}



//-----------------------------------------------------------------------------
// Column Header Datablock Handling
//-----------------------------------------------------------------------------

void GuiModernTextListCtrl::setHeaderProfile(GuiControlProfile *prof)
{
	AssertFatal(prof, "GuiModernTextListCtrl::setHeaderProfile: invalid profile");

	if(prof == mHeaderProfile)
		return;

	bool skipAwaken = false;

	if(mHeaderProfile == NULL)
		skipAwaken = true;

	if(mAwake && mHeaderProfile)
		mHeaderProfile->decRefCount();

	// Clear the delete notification we previously set up
	if(mHeaderProfile)
		clearNotify(mHeaderProfile);

	mHeaderProfile = prof;
	if(mAwake)
	{
		mHeaderProfile->incRefCount();

		mThemedHeaderCell = false;
		if(mHeaderProfile->mUseBitmapArray)
			mThemedHeaderCell = (mHeaderProfile->constructBitmapArray() >= 36);
	}

	// Make sure that the new profile will notify us when it is deleted
	if(mHeaderProfile)
		deleteNotify(mHeaderProfile);

	// force an update when the profile is changed
	if(mAwake && !skipAwaken)
	{
		sleep();

//		if(!Sim::isShuttingDown())
			awaken();
	}
}

bool GuiModernTextListCtrl::setHeaderProfileProt(void *object, const char *data)
{
	GuiModernTextListCtrl	*ctrl = static_cast<GuiModernTextListCtrl*>( object );
	GuiControlProfile		*prof = dynamic_cast<GuiControlProfile*>( Sim::findObject( data ) );
	if(prof == NULL)
		return false;

	// filter through our setter, for consistency
	ctrl->setHeaderProfile(prof);

	// ask the console not to set the data, because we've already set it
	return false;
}


//-----------------------------------------------------------------------------
// Column Sort Indicator Datablock Handling
//-----------------------------------------------------------------------------

void GuiModernTextListCtrl::setSortProfile(GuiControlProfile *prof)
{
	AssertFatal(prof, "GuiModernTextListCtrl::setSortProfile: invalid profile");

	if(prof == mSortProfile)
		return;

	bool skipAwaken = false;

	if(mSortProfile == NULL)
		skipAwaken = true;

	if(mAwake && mSortProfile)
		mSortProfile->decRefCount();

	// Clear the delete notification we previously set up
	if(mSortProfile)
		clearNotify(mSortProfile);

	mSortProfile = prof;
	if(mAwake)
		mSortProfile->incRefCount();

	// Make sure that the new profile will notify us when it is deleted
	if(mSortProfile)
		deleteNotify(mSortProfile);

	// force an update when the profile is changed
	if(mAwake && !skipAwaken)
	{
		sleep();

//		if(!Sim::isShuttingDown())
			awaken();
	}

	mThemedSortCell = mSortProfile->mTextureObject;
}

bool GuiModernTextListCtrl::setSortProfileProt(void *object, const char *data)
{
	GuiModernTextListCtrl	*ctrl = static_cast<GuiModernTextListCtrl*>( object );
	GuiControlProfile		*prof = dynamic_cast<GuiControlProfile*>( Sim::findObject( data ) );
	if(prof == NULL)
		return false;

	// filter through our setter, for consistency
	ctrl->setSortProfile(prof);

	// ask the console not to set the data, because we've already set it
	return false;
}


void GuiModernTextListCtrl::onDeleteNotify(SimObject *object)
{
	if(object == mHeaderProfile)
	{
		GuiControlProfile* profile;
		Sim::findObject("GuiDefaultProfile", profile);

		if(profile == mHeaderProfile)
			mHeaderProfile = NULL;
		else
			setHeaderProfile(profile);

		return;
	}
	if(object == mSortProfile)
	{
		GuiControlProfile* profile;
		Sim::findObject("GuiDefaultProfile", profile);

		if(profile == mSortProfile)
			mSortProfile = NULL;
		else
			setSortProfile(profile);

		return;
	}

	Parent::onDeleteNotify(object);
}


//-----------------------------------------------------------------------------
// Class Specific Utility Functions
//-----------------------------------------------------------------------------

bool GuiModernTextListCtrl::getColumnByIndex(U32 index, tColItem *&col)
{
	if(index >= (U32)mColumns.size())
		return false;

	col = &mColumns[index];
	return true;
}

bool GuiModernTextListCtrl::getColumnByKey(S32 key, tColItem *&col)
{
	tviColumns it;

	// find column associated with key
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		if(it->key == key)
		{
			col = &it[0];
			return true;
		}

		// next
		it++;
	}

	return false;
}

U32 GuiModernTextListCtrl::getColumnIndexByPointer(tColItem *&col)
{
	tviColumns it;
	U32 pos = 0, index = eGUI_InvalidIndex;

	// find column associated with key
	it = mColumns.begin(); pos = 0;
	while(it != mColumns.end())
	{
		if(&it[0] == col)
		{
			index = pos;
			break;
		}

		// next
		it++;
		pos++;
	}

	return index;
}

bool GuiModernTextListCtrl::getRowByIndex(U32 index, tRowItem *&row)
{
	if(index >= (U32)mSortedRows.size())
		return false;

	row = mSortedRows[index];
	return true;
}

bool GuiModernTextListCtrl::getRowByKey(S32 key, tRowItem *&row)
{
	tviRows it;

	// find row associated with key
	it = mRows.begin();
	while(it != mRows.end())
	{
		if(it->key == key)
		{
			row = &*it;
			return true;
		}

		// next
		it++;
	}

	return false;
}

bool GuiModernTextListCtrl::getSortRuleByIndex(U32 index, tSortRule *&rule)
{
	if(index >= (U32)mSortRules.size())
		return false;

	rule = &mSortRules[index];
	return true;
}

bool GuiModernTextListCtrl::getSortRuleByKey(S32 key, tSortRule *&rule)
{
	tviSortRules it;

	// find sort order rule associated with a column key
	it = mSortRules.begin();
	while(it != mSortRules.end())
	{
		if(it->key == key)
		{
			rule = &it[0];
			return true;
		}

		// next
		it++;
	}

	return false;
}

void GuiModernTextListCtrl::parseColumnTextFlags(tColItem &col, const char *flags, bool init)
{
	enum
	{
		ColumnFlagText_Left = 1,
		ColumnFlagText_Right,
		ColumnFlagText_Center,
		ColumnFlagText_Numeric,
		ColumnFlagText_Active,
		ColumnFlagText_HeaderIcon,
		ColumnFlagText_HeaderText,
		ColumnFlagText_Show,
		ColumnFlagText_Hide,
		ColumnFlagText_Visible,
		ColumnFlagText_Invisible,

		ColumnFlagText_End
	};
	struct tColumnFlags
	{
		S32				id;
		const char		*flag;
	};
	static tColumnFlags textFlags[] =
	{
		{ ColumnFlagText_Left,			"left" },
		{ ColumnFlagText_Right,			"right" },
		{ ColumnFlagText_Center,		"center" },
		{ ColumnFlagText_Numeric,		"numeric" },
		{ ColumnFlagText_Active,		"active" },
		{ ColumnFlagText_HeaderIcon,	"headericon" },
		{ ColumnFlagText_HeaderText,	"headertext" },
		{ ColumnFlagText_Show,			"show" },
		{ ColumnFlagText_Hide,			"hide" },
		{ ColumnFlagText_Visible,		"visible" },
		{ ColumnFlagText_Invisible,		"invisible" },
	};
	U32 len, pos, i;
	const char *word;
	bool enabler;


	if(init)
	{
		// initially set the default column type and text content alignment
		col.align	= Column_Align_Left;
		col.type	= Column_Type_Text;
		col.flags	= Column_Flags_Active | Column_Flags_Visible;
	}

	// parse the text flag words
	while(flags && *flags)
	{
		// find whitespace char after word
		pos = dStrcspn(flags, " \t");

		// skip over empty word
		if(pos == 0)
		{
			flags++;
			continue;
		}

		// save word length, position, and bump up flags string
		len   =  pos;
		word  =  flags;
		flags += pos;

		// detect not symbol for disabler
		enabler = true;
		if(*word == '!')
		{
			word++;
			len--;
			enabler = false;
		}

		// iterate through the registered flag text words to find a match
		for(pos=0, i=0; i<(ColumnFlagText_End -1); i++)
		{
			// make sure string lengths match
			if(len != dStrlen(textFlags[i].flag))
				continue;

			// case-insensitive string comparison
			if(dStrnicmp(word, textFlags[i].flag, len))
				continue;

			// flag word match found
			pos = textFlags[i].id;
			break;
		}

		// skip text flag word when not recognized
		if(pos == 0)
		{
			// could possibly report unrecognized flag words here
			continue;
		}

		// process the flag
		switch (pos)
		{
			case ColumnFlagText_Left:			col.align = Column_Align_Left; break;
			case ColumnFlagText_Numeric:
			case ColumnFlagText_Right:			col.align = Column_Align_Right; break;
			case ColumnFlagText_Center:			col.align = Column_Align_Center; break;

			case ColumnFlagText_Active:
			{
				if(enabler)
					col.flags |= Column_Flags_Active;
				else
					col.flags &= ~Column_Flags_Active;
				break;
			}

			case ColumnFlagText_HeaderIcon:
				enabler = !enabler;

			case ColumnFlagText_HeaderText:
			{
				col.type = (enabler)? Column_Type_Text:Column_Type_Image;
				break;
			}

			case ColumnFlagText_Hide:
			case ColumnFlagText_Invisible:
				enabler = !enabler;

			case ColumnFlagText_Show:
			case ColumnFlagText_Visible:
			{
				if(enabler)
					col.flags |= Column_Flags_Visible;
				else
					col.flags &= ~Column_Flags_Visible;
				break;
			}
		}
	}

	// done
}

void GuiModernTextListCtrl::parseRowTextFlags(tRowItem &row, const char *flags, bool init)
{
	enum
	{
		RowFlagText_Active = 1,
		RowFlagText_Selectable,

		RowFlagText_End
	};
	struct tRowFlags
	{
		S32				id;
		const char		*flag;
	};
	static tRowFlags textFlags[] =
	{
		{ RowFlagText_Active,			"active" },
		{ RowFlagText_Selectable,		"selectable" }
	};
	U32 len, pos, i;
	const char *word;
	bool enabler;


	if(init)
	{
		// initially set the default row flags
		row.flags	= Column_Flags_Active | Row_Flags_Selectable;
	}

	// parse the text flag words
	while(flags && *flags)
	{
		// find whitespace char after word
		pos = dStrcspn(flags, " \t");

		// skip over empty word
		if(pos == 0)
		{
			flags++;
			continue;
		}

		// save word length, position, and bump up flags string
		len   =  pos;
		word  =  flags;
		flags += pos;

		// detect not symbol for disabler
		enabler = true;
		if(*word == '!')
		{
			word++;
			len--;
			enabler = false;
		}

		// iterate through the registered flag text words to find a match
		for(pos=0, i=0; i<(RowFlagText_End -1); i++)
		{
			// make sure string lengths match
			if(len != dStrlen(textFlags[i].flag))
				continue;

			// case-insensitive string comparison
			if(dStrnicmp(word, textFlags[i].flag, len))
				continue;

			// flag word match found
			pos = textFlags[i].id;
			break;
		}

		// skip text flag word when not recognized
		if(pos == 0)
		{
			// could possibly report unrecognized flag words here
			continue;
		}

		// process the flag
		switch (pos)
		{
			case RowFlagText_Active:
			{
				if(enabler)
					row.flags |= Row_Flags_Active;
				else
					row.flags &= ~Row_Flags_Active;
				break;
			}

			case RowFlagText_Selectable:
			{
				if(enabler)
					row.flags |= Row_Flags_Selectable;
				else
					row.flags &= ~Row_Flags_Selectable;
				break;
			}
		}
	}

	// done
}

bool GuiModernTextListCtrl::hitTestColumn(Point2I pt, tColItem *&column, U32 &indexPos)
{
	tviColumns	it;
	RectI		rectHeaders;
	U32			index = 0;


	// set default results to not found
	column		= NULL;
	indexPos	= eGUI_InvalidIndex;

	// set the rectangle to cover the entire row's columns
	rectHeaders = RectI(0, pt.y, mCellSize.x, mHeaderDim.y);

	// iterate through all column headers
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		// skip invisible columns
		if(!(it->flags & Column_Flags_Visible))
			goto MoveOnToNext;

		rectHeaders.extent.x = it->width;

		// hit test column
		if(rectHeaders.pointInRect(pt))
		{
			column		= &it[0];
			indexPos	= index;
			return true;
		}

		// push rectangle forward horizontally
		rectHeaders.point.x += it->width;

MoveOnToNext:
		// next column
		it++; index++;
	}

	return false;
}

S32 GuiModernTextListCtrl::calcRowsWidth(void)
{
	tviColumns it;
	S32 width = 0;

	// calculate the width of the control based on all the column headers
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		if(it->flags & Column_Flags_Visible)
			width += it->width;

		// next
		it++;
	}

	return width;
}

S32 GuiModernTextListCtrl::calcRowsHeight(void)
{
	// calculate the height of the control based on all the rows
	return mCellSize.y * mRows.size();
}

bool GuiModernTextListCtrl::getBitmap(const char *filename, tBitmap *&bitmap)
{
	tviBitmaps		it;
	dsize_t			len;

	// default to failure result
	bitmap = NULL;

	// check to see if we've already got a handle on that bitmap
	it = mBitmaps.begin();
	while(it != mBitmaps.end())
	{
		if((len = dStrlen(it->filename)) != dStrlen(filename) ||
			dStrnicmp(it->filename, filename, len))
		{
			// next
			it++;
			continue;
		}

		// found it
		bitmap = &it[0];
		return true;
	}


	tBitmap tex;

	tex.texture.set(filename, &GFXDefaultPersistentProfile, avar("%s() - tex.bitmapObject (line %d)", __FUNCTION__, __LINE__));
	if(!(bool)tex.texture)
		return false; // failed to create texture

	tex.filename	= dStrdup(filename);
	tex.extent.x	= tex.texture.getWidth();
	tex.extent.y	= tex.texture.getHeight();
	tex.refCount	= 1;

	mBitmaps.push_back(tex);
	bitmap = &mBitmaps[mBitmaps.size() -1];

	return true;
}

void GuiModernTextListCtrl::clearBitmaps()
{
	tviBitmaps		it;

	// destroy bitmaps
	it = mBitmaps.begin();
	while(it != mBitmaps.end())
	{
		// deallocate memory
		dFree(it->filename);

		// next
		it++;
	}

	// erase the vector
	mBitmaps.clear();
}


//-----------------------------------------------------------------------------
// Methods
//-----------------------------------------------------------------------------

U32 GuiModernTextListCtrl::addColumn(S32 key, const char *name, U32 width, U32 minWidth, U32 maxWidth, const char *flags)
{
	tColItem col;

	// create a new column
	col.column		= mColumns.size();
	col.key			= key;
	col.name		= dStrdup(name);
	col.width		= width;
	col.minWidth	= minWidth;
	col.maxWidth	= maxWidth;
	col.sort		= 0;
	col.sortPos		= 0;

	// parse and process the provided column flags
	parseColumnTextFlags(col, flags);

	// store the new column
	mColumns.push_back(col);

	// update control size
	if(mResizeOnChange)
		updateSize();

	return col.column;
}

U32 GuiModernTextListCtrl::addRow(S32 key, const char *text, U32 index, const char *flags)
{
	tRowItem row, *pRow;

	// create a new row
	index			= mRows.size();
	row.key			= key;
	row.text		= dStrdup(text);
	row.flags		= 0;

#if defined(MODERNLIST_SORT_STRIPONCE)
	char *stripped;

	// pre-strip the markup text for column sorting use
	// and this is intentionally left alone even after
	// markup is disabled via mUseMarkup, so Don't touch!
	stripMarkupTextFields(stripped, text, true);

	row.sortText	= dStrdup(stripped);
#endif // MODERNLIST_SORT_STRIPONCE

	// parse and process the provided row flags
	parseRowTextFlags(row, flags);

	// store the new row
	mRows.push_back(row);
	pRow = &*mRows.rbegin();

	// store the new sorted row pointer
	if(index < mSortedRows.size())
	{
		mSortedRows.insert(mSortedRows.begin() + index, pRow);
	} else
	{
		mSortedRows.push_back(pRow);
		index = mSortedRows.size() -1;
	}

	// update control size
	if(mResizeOnChange)
		updateSize();

	return index;
}

void GuiModernTextListCtrl::removeRow(U32 index)
{
	tviRows		it;
	tRowItem	*row;


	// ignore invalid index position
	if(index >= mSortedRows.size())
		return;

	// free any allocated memory for the row
	row = mSortedRows[index];
	dFree(row->text);
#if defined(MODERNLIST_SORT_STRIPONCE)
	dFree(row->sortText);
#endif // MODERNLIST_SORT_STRIPONCE

	// clear deferred draw row
	if(mDeferred.row == row)
		mDeferred.row = NULL;

	// delete the row pointer
	mSortedRows.erase(mSortedRows.begin() + index);

	// locate the real row item and delete it too
	it = mRows.begin();
	while(it != mRows.end())
	{
		if(&*it == row)
		{
			mRows.erase(it);
			break;
		}
	}

	// make sure row selection is valid
	if(mSelectedCell.y >= mSortedRows.size())
		mSelectedCell.y = mSortedRows.size() -1;

	// update control size
	updateSize();

	// done
}

void GuiModernTextListCtrl::clearColumns(void)
{
	tviColumns it;


	// clear deferred drawing column
	mDeferred.column = NULL;

	// destroy the columns
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		// unallocate the column name string
		dFree(it->name);

		// next
		it++;
	}

	// erase the vector
	mColumns.clear();

	// update control size
	updateSize();
}

void GuiModernTextListCtrl::clearRows(void)
{
	tviRows it;


	// clear deferred drawing row
	mDeferred.row = NULL;

	// destroy the rows
	it = mRows.begin();
	while(it != mRows.end())
	{
		// unallocate the row text string
		dFree(it->text);
#if defined(MODERNLIST_SORT_STRIPONCE)
		dFree(it->sortText);
#endif // MODERNLIST_SORT_STRIPONCE

		// next
		it++;
	}

	// erase the list and pointer vector
	mRows.clear();
	mSortedRows.clear();

	// clear cached bitmaps
	clearBitmaps();

	// update control size
	updateSize();
}

void GuiModernTextListCtrl::clearSortOrders(void)
{
	// destroy the column sort orders by just erasing the vector
	mSortRules.clear();

	// update columns sort indicator
	sortColumns();
}

void GuiModernTextListCtrl::getColumnFlags(U32 index, char *flags)
{
	tColItem *col;

	// default to empty string
	*flags = 0;

	if(!getColumnByIndex(index, col))
		return;

	switch (col->align)
	{
		default:
		case Column_Align_Left:		dStrcat(flags, "left "); break;
		case Column_Align_Right:	dStrcat(flags, "right "); break;
		case Column_Align_Center:	dStrcat(flags, "center "); break;
	}

	if(col->type == Column_Type_Image)
		dStrcat(flags, "headericon ");
	else
		dStrcat(flags, "headertext ");

	if(!(col->flags & Column_Flags_Active))
		dStrcat(flags, "!");
	dStrcat(flags, "active ");

	if(!(col->flags & Column_Flags_Visible))
		dStrcat(flags, "!");
	dStrcat(flags, "show");

}

const char* GuiModernTextListCtrl::getColumnName(U32 index)
{
	tColItem *col;

	if(getColumnByIndex(index, col))
		return col->name;

	return NULL;
}

S32 GuiModernTextListCtrl::getColumnKey(U32 index)
{
	tColItem *col;

	if(getColumnByIndex(index, col))
		return col->key;

	return eGUI_InvalidIndex;
}

U32 GuiModernTextListCtrl::getColumnWidth(U32 index)
{
	tColItem *col;

	if(getColumnByIndex(index, col))
		return col->width;

	return (U32)eGUI_InvalidIndex;
}

U32 GuiModernTextListCtrl::getColumnIndex(S32 key)
{
	tviColumns it;
	U32 index = 0;

	// find column associated with key
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		if(it->key == key)
		{
			// return index position
			return index;
		}

		// next
		it++;
		index++;
	}

	return (U32)eGUI_InvalidIndex;
}

U32 GuiModernTextListCtrl::getColumnCount(void)
{
	return mColumns.size();
}

U32 GuiModernTextListCtrl::getSortOrderCount(void)
{
	return mSortRules.size();
}

const GuiModernTextListCtrl::tSortRule* GuiModernTextListCtrl::getSortOrder(U32 index)
{
	tSortRule *rule;

	if(getSortRuleByIndex(index, rule))
		return rule;

	return NULL;
}

U32 GuiModernTextListCtrl::getSelectedRow(void)
{
	return mSelectedCell.y;
}

U32 GuiModernTextListCtrl::getRowCount(void)
{
	return mRows.size();
}

void GuiModernTextListCtrl::getRowFlags(U32 index, char *flags)
{
	tRowItem *row;

	// default to empty string
	*flags = 0;

	if(!getRowByIndex(index, row))
		return;

	if(!(row->flags & Row_Flags_Active)) dStrcat(flags, "!");
	dStrcat(flags, "active ");

	if(!(row->flags & Row_Flags_Selectable)) dStrcat(flags, "!");
	dStrcat(flags, "selectable");

}

const char* GuiModernTextListCtrl::getRowText(U32 index)
{
	tRowItem *row;

	if(getRowByIndex(index, row))
		return row->text;

	return NULL;
}

S32 GuiModernTextListCtrl::getRowKey(U32 index)
{
	tRowItem *row;

	if(getRowByIndex(index, row))
		return row->key;

	return eGUI_InvalidIndex;
}

U32 GuiModernTextListCtrl::getRowIndex(S32 key)
{
	tviSortedRows	it;
	U32				index = 0;

	// find column associated with key
	it = mSortedRows.begin();
	while(it != mSortedRows.end())
	{
		if((*it)->key == key)
		{
			// return index position
			return index;
		}

		// next
		it++;
		index++;
	}

	return eGUI_InvalidIndex;
}

void GuiModernTextListCtrl::scrollVisible(U32 index)
{
	if(index < mRows.size())
		this->scrollCellVisible(Point2I(0, index));
}

void GuiModernTextListCtrl::setSelectedRow(U32 index)
{
	if(index == eGUI_InvalidIndex)
		mSelectedCell.y = -1;
	else if(index < mRows.size())
		mSelectedCell.y = index;
}

void GuiModernTextListCtrl::setSortOrder(const char *rule)
{
	S32 key, order, first=0;

	if(dSscanf(rule, "%ld %ld %ld", &key, &order, &first) > 1)
		setColumnSortOrder(key, order, first);

}

void GuiModernTextListCtrl::setColumnFlags(U32 index, const char *flags)
{
	tColItem *col;

	if(!getColumnByIndex(index, col))
		return;

	// parse and process the provided column flags
	parseColumnTextFlags(*col, flags, false);

	// update control size
	if(mResizeOnChange)
		updateSize();
}

void GuiModernTextListCtrl::setColumnSortOrder(S32 key, S32 order, bool pushToFirst)
{
	tSortRule		*rule, arule;
	tviSortRules	it;


	if(getSortRuleByKey(key, rule))
	{
		rule->order = order;

		if(!pushToFirst)
			return;

		arule = *rule;

		it = mSortRules.begin();
		while(it != mSortRules.end())
		{
			if(&it[0] != rule)
			{
				// next
				it++;
				continue;
			}

			mSortRules.erase(it);
			mSortRules.insert(mSortRules.begin(), arule);
			break;
		}

		return;
	}

	arule.key	= key;
	arule.order	= order;

	if(pushToFirst)
		mSortRules.insert(mSortRules.begin(), arule);
	else
		mSortRules.push_back(arule);

}

void GuiModernTextListCtrl::setColumnWidth(U32 index, U32 width)
{
	tColItem *col;

	if(!getColumnByIndex(index, col))
		return;

	// enforce range limits
	if(width < col->minWidth)	width = col->minWidth;
	if(width > col->maxWidth)	width = col->maxWidth;

	// don't bother if the same
	if(width == col->width)
		return;

	col->width = width;

	// update control size
	if(mResizeOnChange)
		updateSize();
}

void GuiModernTextListCtrl::setRowFlags(U32 index, const char *flags)
{
	tRowItem *row;
	
	if(!getRowByIndex(index, row))
		return;

	// parse and process the provided row flags
	parseRowTextFlags(*row, flags, false);

}

void GuiModernTextListCtrl::setRowText(U32 index, const char *text)
{
	tRowItem *row;
	

	if(!getRowByIndex(index, row))
		return;

	// free existing text string and allocate another one for the new text string
	dFree(row->text);
	row->text = dStrdup(text);

#if defined(MODERNLIST_SORT_STRIPONCE)
	char *stripped;
	
	// pre-strip the markup text for column sorting use
	stripMarkupTextFields(stripped, text, true);

	dFree(row->sortText);
	row->sortText = dStrdup(stripped);
#endif // MODERNLIST_SORT_STRIPONCE

}

void GuiModernTextListCtrl::sortColumns(void)
{
	if(mSortRules.size())
	{
		tviSortRules	it;
		tColItem		*col;
		U8				pos = 0;

		// sort rows based on current column sort order rules
		mSortManager->setMarkupEnable(mUseMarkup);
		std::sort(mSortedRows.begin(), mSortedRows.end(), *mSortManager);
		
		// update each column's sort indicator
		it = mSortRules.begin();
		while(it != mSortRules.end())
		{
			if(getColumnByKey(it->key, col))
			{
				col->sort		= it->order +1;
				col->sortPos	= pos;
			}

			// next
			it++;
			pos++;
		}

		return;
	}

	// there aren't any column sort order rules, clear the sort indicator on all columns
	tviColumns it = mColumns.begin();
	while(it != mColumns.end())
	{
		it->sort = 0;

		// next
		it++;
	}

}

void GuiModernTextListCtrl::moveColumn(S32 key, U32 index)
{
	tviColumns	it;
	tColItem	col;


	// check to see if the column index position is even possible
	if(index >= mColumns.size())
		return; // not possible

	// find column associated with key
	it = mColumns.begin();
	while(it != mColumns.end())
	{
		if(it->key != key)
		{
			// next
			it++;
			continue;
		}

		// keep a copy
		col = it[0];

		// delete the column
		mColumns.erase(it);

		// re-insert column at requested index position
		mColumns.insert(mColumns.begin() + index, col);
		break;
	}

	// done
}

void GuiModernTextListCtrl::moveRow(S32 key, U32 index)
{
	tviSortedRows	it;
	tRowItem		*row;


	// check to see if the row index position is even possible
	if(index >= mSortedRows.size())
		return; // not possible

	// find row associated with key
	it = mSortedRows.begin();
	while(it != mSortedRows.end())
	{
		if((*it)->key != key)
		{
			// next
			it++;
			continue;
		}

		// keep a copy
		row = *it;

		// delete the row
		mSortedRows.erase(it);

		// re-insert row at requested index position
		mSortedRows.insert(mSortedRows.begin() + index, row);
		break;
	}

	// done
}

const char *GuiModernTextListCtrl::getScriptValue()
{
//	return getRowText( getSelectedRow() );
	return "";
}

void GuiModernTextListCtrl::setScriptValue(const char *val)
{
/*
	S32 e = findEntryByText(val);
	if(e == -1)
		setSelectedCell(Point2I(-1, -1));
	else
		setSelectedCell(Point2I(0, e));
*/
}

void GuiModernTextListCtrl::setSize(Point2I newSize)
{
//	mSize = newSize;

	if (bool( mFont ))
	{
		mCellSize.y = mFont->getHeight();
	}

	mHeaderDim.y	 = mCellSize.y		+ mHeaderTextPadding.y;
	mCellSize.x		 = calcRowsWidth();
	mCellSize.y		+= mRowTextPadding.y;

	mSize = Point2I(1, mRows.size() * mCellSize.y);

	Point2I newExtent(mCellSize.x + mHeaderDim.x, mRows.size() * mCellSize.y + mHeaderDim.y);
	setExtent(newExtent);
}


//=============================================================================
// Console Methods
//=============================================================================

#define	GetMethodArg(_n)	argv[_n + 2]
#define	ArgCount(_n)		(_n + 2)

//DefineEngineMethod( GuiModernTextListCtrl, addColumn, S32, (S32 key, const char *name, U32 width, U32 minWidth, U32 maxWidth, const char *flags), (0, "", 50, 10, 250, ""),
//	"\n")
ConsoleMethod( GuiModernTextListCtrl, addColumn, S32, ArgCount(2), ArgCount(6), "(S32 key, string name, U32 width, U32 minWidth, U32 maxWidth, string flags)")
{
	S32			key			= dAtoi(GetMethodArg(0));
	const char	*name		= GetMethodArg(1);
	U32			width		=  50;
	U32			minWidth	=  10;
	U32			maxWidth	= 250;
	const char *flags		= "";


	if(argc > ArgCount(2))	width		= dAtoui(GetMethodArg(2));
	if(argc > ArgCount(3))	minWidth	= dAtoui(GetMethodArg(3));
	if(argc > ArgCount(4))	maxWidth	= dAtoui(GetMethodArg(4));
	if(argc > ArgCount(5))	flags		= GetMethodArg(5);

	return object->addColumn(key, name, width, minWidth, maxWidth, flags);
}

//DefineEngineMethod( GuiModernTextListCtrl, addRow, S32, (S32 key, const char *text, U32 index, const char *flags), (0, "", -1, "") ,
//	"\n")
ConsoleMethod( GuiModernTextListCtrl, addRow, S32, ArgCount(2), ArgCount(4), "(S32 key, string text, U32 index, string flags)")
{
	S32			key			= dAtoi(GetMethodArg(0));
	const char	*text		= GetMethodArg(1);
	U32			index		=  50;
	const char *flags		= "";


	if(argc > ArgCount(2))	index		= dAtoui(GetMethodArg(2));
	if(argc > ArgCount(3))	flags		= GetMethodArg(3);

	return object->addRow(key, text, index, flags);
}

//DefineEngineMethod( GuiModernTextListCtrl, removeRow, void, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, removeRow, void, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	object->removeRow(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, clearColumns, void, (), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, clearColumns, void, ArgCount(0), ArgCount(0), "()")
{
	object->clearColumns();
}

//DefineEngineMethod( GuiModernTextListCtrl, clearRows, void, (), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, clearRows, void, ArgCount(0), ArgCount(0), "()")
{
	object->clearRows();
}

//DefineEngineMethod( GuiModernTextListCtrl, clearSortOrders, void, (), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, clearSortOrders, void, ArgCount(0), ArgCount(0), "()")
{
	object->clearSortOrders();
}

//DefineEngineMethod( GuiModernTextListCtrl, getColumnFlags, const char*, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getColumnFlags, const char*, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	U32 buffSize = MAXFLAGSTEXTLEN;
	char *str = Con::getReturnBuffer(buffSize);
	*str = 0;

	object->getColumnFlags(index, str);
	return str;
}

//DefineEngineMethod( GuiModernTextListCtrl, getColumnName, const char*, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getColumnName, const char*, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	return object->getColumnName(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, getColumnKey, S32, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getColumnKey, S32, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	return object->getColumnKey(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, getColumnWidth, S32, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getColumnWidth, S32, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	return object->getColumnWidth(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, getColumnIndex, S32, (S32 key), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getColumnIndex, S32, ArgCount(1), ArgCount(1), "(S32 key)")
{
	S32			key		=  dAtoi(GetMethodArg(0));

	return object->getColumnIndex(key);
}

//DefineEngineMethod( GuiModernTextListCtrl, getColumnCount, S32, (), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getColumnCount, S32, ArgCount(0), ArgCount(0), "()")
{
	return object->getColumnCount();
}

//DefineEngineMethod( GuiModernTextListCtrl, getSortOrderCount, S32, (), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getSortOrderCount, S32, ArgCount(0), ArgCount(0), "()")
{
	return object->getSortOrderCount();
}

// const GuiModernTextListCtrl::tSortRule *getSortOrder(U32 index);
//DefineEngineMethod( GuiModernTextListCtrl, getSortOrder, const char*, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getSortOrder, const char*, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	const GuiModernTextListCtrl::tSortRule *rule;
	U32 buffSize = 32;
	char *str = Con::getReturnBuffer(buffSize);
	*str = 0;

	rule = object->getSortOrder(index);
	if(rule)
		dSprintf(str, buffSize, "%ld %ld", rule->key, rule->order);

	return str;
}

//DefineEngineMethod( GuiModernTextListCtrl, getSelectedRow, S32, (),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getSelectedRow, S32, ArgCount(0), ArgCount(0), "()")
{
	return object->getSelectedRow();
}

//DefineEngineMethod( GuiModernTextListCtrl, getRowCount, S32, (),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getRowCount, S32, ArgCount(0), ArgCount(0), "()")
{
	return object->getRowCount();
}

//DefineEngineMethod( GuiModernTextListCtrl, getRowFlags, const char*, (U32 index), ,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getRowFlags, const char*, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	U32 buffSize = MAXFLAGSTEXTLEN;
	char *str = Con::getReturnBuffer(buffSize);
	*str = 0;

	object->getRowFlags(index, str);
	return str;
}

//DefineEngineMethod( GuiModernTextListCtrl, getRowText, const char*, (U32 index),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getRowText, const char*, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	return object->getRowText(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, getRowKey, S32, (U32 index),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getRowKey, S32, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	return object->getRowKey(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, getRowIndex, S32, (S32 key),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, getRowIndex, S32, ArgCount(1), ArgCount(1), "(U32 index)")
{
	S32			key		=  dAtoi(GetMethodArg(0));

	return object->getRowIndex(key);
}

//DefineEngineMethod( GuiModernTextListCtrl, scrollVisible, void, (U32 rowNum),,
//	"@brief Scroll so the specified row is visible\n\n"
//	"@param rowNum Row number to make visible\n"
//	"@tsexample\n"
//	"// Define the row number to make visible\n"
//	"%rowNum = \"4\";\n\n"
//	"// Inform the GuiTextListCtrl control to scroll the list so the defined rowNum is visible.\n"
//	"%thisGuiTextListCtrl.scrollVisible(%rowNum);\n"
//	"@endtsexample\n\n"
//	"@see GuiControl")
ConsoleMethod( GuiModernTextListCtrl, scrollVisible, void, ArgCount(1), ArgCount(1), "(U32 rowNum)")
{
	S32			rowNum	=  dAtoui(GetMethodArg(0));

	object->scrollCellVisible(Point2I(0, rowNum));
}

//DefineEngineMethod( GuiModernTextListCtrl, setSelectedRow, void, (U32 index),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setSelectedRow, void, ArgCount(1), ArgCount(1), "(U32 index)")
{
	U32			index		=  dAtoui(GetMethodArg(0));

	object->setSelectedRow(index);
}

//DefineEngineMethod( GuiModernTextListCtrl, setSortOrder, void, (const char *rule),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setSortOrder, void, ArgCount(1), ArgCount(1), "(string rule)")
{
	const char	*rule		=  GetMethodArg(0);

	object->setSortOrder(rule);
}

//DefineEngineMethod( GuiModernTextListCtrl, setColumnFlags, void, (U32 index, const char *flags),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setColumnFlags, void, ArgCount(2), ArgCount(2), "(U32 index, string flags)")
{
	U32			index		=  dAtoui(GetMethodArg(0));
	const char	*flags		=  GetMethodArg(1);

	object->setColumnFlags(index, flags);
}

//DefineEngineMethod( GuiModernTextListCtrl, setColumnSortOrder, void, (S32 key, S32 order, bool pushToFirst), (false),
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setColumnSortOrder, void, ArgCount(2), ArgCount(3), "(S32 key, S32 order, bool pushToFirst)")
{
	S32			key			=  dAtoi(GetMethodArg(0));
	S32			order		=  dAtoi(GetMethodArg(1));
	bool		pushToFirst	= false;

	if(argc > ArgCount(2))	pushToFirst	= dAtob(GetMethodArg(2));

	object->setColumnSortOrder(key, order, pushToFirst);
}

//DefineEngineMethod( GuiModernTextListCtrl, setColumnWidth, void, (U32 index, U32 width),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setColumnWidth, void, ArgCount(2), ArgCount(2), "(U32 index, U32 width)")
{
	U32			index		=  dAtoui(GetMethodArg(0));
	U32			width		=  dAtoui(GetMethodArg(1));

	object->setColumnWidth(index, width);
}

//DefineEngineMethod( GuiModernTextListCtrl, setRowFlags, void, (U32 index, const char *flags),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setRowFlags, void, ArgCount(2), ArgCount(2), "(U32 index, string flags)")
{
	S32			index		=  dAtoui(GetMethodArg(0));
	const char	*flags		= GetMethodArg(1);

	object->setRowFlags(index, flags);
}

//DefineEngineMethod( GuiModernTextListCtrl, setRowText, void, (U32 index, const char *text),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, setRowText, void, ArgCount(2), ArgCount(2), "(U32 index, string text)")
{
	S32			index		=  dAtoui(GetMethodArg(0));
	const char	*text		= GetMethodArg(1);

	object->setRowText(index, text);
}

//DefineEngineMethod( GuiModernTextListCtrl, setHeaderProfile, void, (GuiControlProfile* profile),,
//   "Set the header profile for the control to use.\n"
//   "The header profile used by a control determines a great part of its behavior and appearance for the column headers.\n"
//   "@param profile The new header profile the control should use.\n"
//   "@ref GuiControl_Profiles" )
ConsoleMethod( GuiModernTextListCtrl, setHeaderProfile, void, ArgCount(1), ArgCount(1), "(GuiControlProfile profile)")
{
	GuiControlProfile * profile;

	if(Sim::findObject(GetMethodArg(0), profile))
		object->setHeaderProfile(profile);
}

//DefineEngineMethod( GuiModernTextListCtrl, setSortProfile, void, (GuiControlProfile* profile),,
//   "Set the column header sort indicator profile for the control to use.\n"
//   "The header profile used by a control determines a great part of its behavior and appearance for the column header sort indicator.\n"
//   "@param profile The new header profile the control should use.\n"
//   "@ref GuiControl_Profiles" )
ConsoleMethod( GuiModernTextListCtrl, setSortProfile, void, ArgCount(1), ArgCount(1), "(GuiControlProfile profile)")
{
	GuiControlProfile * profile;

	if(Sim::findObject(GetMethodArg(0), profile))
		object->setSortProfile(profile);
}

//DefineEngineMethod( GuiModernTextListCtrl, sortColumns, void, (),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, sortColumns, void, ArgCount(0), ArgCount(0), "()")
{
	object->sortColumns();
}

//DefineEngineMethod( GuiModernTextListCtrl, moveColumn, void, (S32 key, U32 index),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, moveColumn, void, ArgCount(2), ArgCount(2), "(S32 key, U32 index)")
{
	S32			key			=  dAtoi(GetMethodArg(0));
	U32			index		=  dAtoui(GetMethodArg(1));

	object->moveColumn(key, index);
}

//DefineEngineMethod( GuiModernTextListCtrl, moveRow, void, (S32 key, U32 index),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, moveRow, void, ArgCount(2), ArgCount(2), "(S32 key, U32 index)")
{
	S32			key			=  dAtoi(GetMethodArg(0));
	U32			index		=  dAtoui(GetMethodArg(1));

	object->moveRow(key, index);
}

//DefineEngineMethod( GuiModernTextListCtrl, stripMarkup, const char *, (const char *text),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, stripMarkup, const char *, ArgCount(1), ArgCount(1), "(string text)")
{
	static char temp[MAXTEXTLEN];
	object->stripMarkup(temp, GetMethodArg(0));

	char *str = Con::getReturnBuffer(dStrlen(temp) +1);
	dStrcpy(str, temp);

	return str;
}

//DefineEngineMethod( GuiModernTextListCtrl, updateSize, void, (),,
//   "\n")
ConsoleMethod( GuiModernTextListCtrl, updateSize, void, ArgCount(0), ArgCount(0), "()")
{
	object->updateSize();
}



//=============================================================================
// GuiModernTextListCtrl's Sort Comparer handling Class
//=============================================================================

bool GuiModernSort::operator()(const GuiModernTextListCtrl::tRowItem *a, const GuiModernTextListCtrl::tRowItem *b)
{
	GuiModernTextListCtrl::tColItem		*col;
	GuiModernTextListCtrl::tviSortRules	it;

	// iterate through the sort order rules
	it = mRules->begin();
	while(it != mRules->end())
	{
		// get associated column
		if(!mOwner->getColumnByKey(it->key, col))
			break; // abort

#if !defined(MODERNLIST_SORT_STRIPONCE)
		if(mMarkup)
		{
			// get row A's column specific content and strip the content of markup
			getColumnContent(a->text, mTextT, sizeof(mTextT), col->column);
			mOwner->stripMarkupText(mTextA, mTextT, true);

			// get row B's column specific content and strip the content of markup
			getColumnContent(b->text, mTextT, sizeof(mTextT), col->column);
			mOwner->stripMarkupText(mTextB, mTextT, true);
		} else
		{
			// get row A and B's column specific content
			getColumnContent(a->text, mTextA, sizeof(mTextA), col->column);
			getColumnContent(b->text, mTextB, sizeof(mTextB), col->column);
		}

#else // MODERNLIST_SORT_STRIPONCE

		// get A and B's column specific content that has been pre-stripped of markup
		if(mMarkup)
		{
			getColumnContent(a->sortText, mTextA, sizeof(mTextA), col->column);
			getColumnContent(b->sortText, mTextB, sizeof(mTextB), col->column);
		} else
		{
			getColumnContent(a->text, mTextA, sizeof(mTextA), col->column);
			getColumnContent(b->text, mTextB, sizeof(mTextB), col->column);
		}

#endif // MODERNLIST_SORT_STRIPONCE

		switch (dStrnatcasecmp(mTextA, mTextB))
		{
			case  1: return (it->order)? true:false;
			case -1: return (it->order)? false:true;
		}

		// next
		it++;
	}

	return false;
}

