//-----------------------------------------------------------------------------
// Torque
// Copyright GarageGames, LLC 2011
// Copyright Nathan Martin 2011-2013
//-----------------------------------------------------------------------------

#ifndef _GUIMODERNTEXTLISTCTRL_H_
#define _GUIMODERNTEXTLISTCTRL_H_

#ifndef _GUIARRAYCTRL_H_
#include "gui/core/guiArrayCtrl.h"
#endif

#include <vector>
#include <list>


// Use the define below to use code that keeps a dedicated text string buffer of
// stripped markup row content upon adding and setting of rows. Else, the markup
// stripping is performed per stl::sort() comparer function call.
// Advantages:		Noticeable speedup on slower CPUs during sorting of >10k rows.
// Disadvantages:	Nearly if not exactly twice as much memory allocated per row.
#undef MODERNLIST_SORT_STRIPONCE


// Declares the maximum text size minus one byte for NULL of text we'll be able to
// manipulate as per row column content limit.
#define MAXTEXTLEN	4096

// Declares the maximum text size minus one byte for NULL of string buffers that
// will be used to store column header or row flags in textual format.
#define MAXFLAGSTEXTLEN	256

// Declares support for TGEA, tested with v1.8.2 only. Do not define for Torque3D
// as that will lead to engine crashing whenever this control is used.
// Disabled by default for T3D support. change #undef to #define for TGEA.
#define SUPPORT_TGEA



//=============================================================================
// TGEA engineAPI alternative wrapper macros
//=============================================================================
#if defined(SUPPORT_TGEA)

#define DECLARE_CALLBACK(_returntype, _name, _args) \
	_returntype _name##_callback _args

#define IMPLEMENT_CALLBACK(_class, _name, _returntype, _args, _argnames, _helptext) \
	_returntype _class::_name ## _callback _args


#endif // !SUPPORT_TGEA


//=============================================================================
// GuiModernTextListCtrl implementation class
//=============================================================================
class GuiModernSort;

class GuiModernTextListCtrl : public GuiArrayCtrl
{
private:
	typedef GuiArrayCtrl Parent;

public:
	enum
	{
		// column content text alignment
		Column_Align_Left		= 0,		// column text content is left aligned
		Column_Align_Right,					// column text content is right aligned
		Column_Align_Center,				// column text content is center aligned

		// column header label type
		Column_Type_Text		= 0,		// column header will display a text name label
		Column_Type_Image,					// column header will display a bitmap icon in the name label

		// column flags
		Column_Flags_Active		= 0x01,		// column header is in an active/enabled state, else in a inactive/disabled state
		Column_Flags_Visible	= 0x02,		// column header is in a invisible state, else in a hidden state
		Column_Flags_Deferred	= 0x04,		// column header draw state is deferred and will be drawn last (move, glow effects, etc..)
		Column_Flags_Highlight	= 0x10,		// column header is in highlight state, mouse is hovering over it
		Column_Flags_Select		= 0x20,		// column header is in a select/depressed state

											// clear soft or temporary states such as Deferred, Highlight, and Select bits
		Column_Flags_ClearSoft	= Column_Flags_Deferred | Column_Flags_Highlight | Column_Flags_Select,


		// row flags
		Row_Flags_Active		= 0x01,		// row is in an active/enabled state, else in a inactive/disabled state for visual representation
		Row_Flags_Selectable	= 0x02,		// row is user selectable, else it isn't. Note setSelectedRow() ignores this flag bit.

	};
	// header column item data
	struct tColItem
	{
		U32			column;					// the actual (logical) column that this column header is responsible for
		S32			key;					// column key
		char		*name;					// column name
		U32			width;					// current column width
		U32			minWidth;				// minimum column width
		U32			maxWidth;				// maximum column width
		U8			align;					// column text alignment
		U8			type;					// not used at this time
		U8			sort;					// sort order indicator 0: none, 1: ascending, 2: descending
		U8			sortPos;				// sort order position of the column, used for determining sort indicator color
		U8			flags;					// misc. column bit flags
	};
	// sort order rule item data
	struct tSortRule
	{
		S32			key;					// associated key of the column to be sorted
		S32			order;					// sort order: 0 = Ascending, 1 = Descending
	};
	// row item data
	struct tRowItem
	{
		S32					key;			// row key
		char				*text;			// row content
		U32					flags;			// row bit flags
#if defined(MODERNLIST_SORT_STRIPONCE)
		char				*sortText;		// row content preprocessed for column sorting use
#endif // MODERNLIST_SORT_STRIPONCE
	};
	struct tBitmap
	{
		char				*filename;		// resource path to texture
		GFXTexHandle		texture;		// texture handle
		Point2I				extent;			// dimensions of the texture
		U32					refCount;		// reference counter, not really used
	};


	std::vector<tColItem>	mColumns;			// header columns
	std::vector<tSortRule>	mSortRules;			// header column sort order rules
	std::list<tRowItem>		mRows;				// row items
	std::vector<tRowItem*>	mSortedRows;		// sorted row items
	std::vector<tBitmap>	mBitmaps;			// bitmaps
	GuiModernSort			*mSortManager;		// pointer to the modern sort instance


	typedef std::vector<tColItem>::iterator		tviColumns;
	typedef std::vector<tSortRule>::iterator	tviSortRules;
	typedef std::list<tRowItem>::iterator		tviRows;
	typedef std::vector<tRowItem*>::iterator	tviSortedRows;
	typedef std::vector<tBitmap>::iterator		tviBitmaps;

protected:
	friend GuiModernSort;

	enum {
		eGUI_InvalidIndex = -1 //0xFFFFFFFF		// invalid index for either column or row
	};

	bool			mThemedHeaderCell;
	bool			mThemedRowCell;
	bool			mThemedSortCell;
	const GuiEvent	*mGuiEvent;					///< Saved pointer of event during GUI events such as OnMouseDown()

	bool getColumnByIndex(U32 index, tColItem *&col);
	bool getColumnByKey(S32 key, tColItem *&col);
	U32  getColumnIndexByPointer(tColItem *&col);
	bool getRowByIndex(U32 index, tRowItem *&row);
	bool getRowByKey(S32 key, tRowItem *&row);
	bool getSortRuleByIndex(U32 index, tSortRule *&rule);
	bool getSortRuleByKey(S32 key, tSortRule *&rule);
	void parseColumnTextFlags(tColItem &col, const char *flags, bool init = true);
	void parseRowTextFlags(tRowItem &row, const char *flags, bool init = true);
	bool hitTestColumn(Point2I pt, tColItem *&column, U32 &indexPos);
	S32  calcRowsWidth(void);
	S32  calcRowsHeight(void);
	bool getBitmap(const char *filename, tBitmap *&bitmap);
	void clearBitmaps(void);

	Point2I prepTextAlignment(GuiControlProfile *profile, RectI &rectCell, char *text, S32 align, Point2I &padding, S32 additional = 0);
	void prepCellDraw(GuiControlProfile *profile, RectI &rectCell, Point2I &glowOffset, bool active, bool selected, bool mouseOver, bool texture);
	S32  stripMarkupText(char *dstText, const char *srcText, bool keepPrimitives = false, bool preservDest = false);
	void stripMarkupTextFields(char *&dstText, const char *srcText, bool keepPrimitives);
	void drawMarkupText(GuiControlProfile *profile, RectI &rectCell, Point2I &textPos, const char *text);
	void drawColumnSortArrow(tColItem *col, RectI &drawRect, RectI &rectText);


	// exposed control properties
	enum ScrollConst
	{
		UP = 0,
		DOWN = 1
	};
	GuiControlProfile		*mHeaderProfile;			///< The profile for this text list's column headers (data settings that are likely to be shared by multiple guis)
	GuiControlProfile		*mSortProfile;				///< The profile for this text list's column sort indicator (texture image, color shading, etc..)
	Point2I					mHeaderTextPadding;			///< Text padding for column header labels
	Point2I					mRowTextPadding;			///< Text padding for row content
	Point2I					mHeaderGlowOffset;			///< Column header cell texture draw offset
	Point2I					mRowGlowOffset;				///< Row cell texture draw offset
	bool					mAllowColumnResize;			///< Whether or not user is allowed to resize columns using the mouse cursor
	bool					mAllowColumnMove;			///< Whether or not user is allowed to move columns via drag and drop
	bool					mUseMarkup;					///< Whether the control should use or just ignore markup language within row contents
	bool					mResizeOnChange;			///< Whether or not to update the control size for the scroller upon content changes, ex. Add[Row|Column], etc..

	// internal state machine variables
	struct
	{
		tColItem			*column;					///< Column header currently being resized or dragged to another position
		S32					startPos;					///< Horizontal pixel start position of the active column
		bool				active;						///< Column drag and drop or resize action is being actively done flag
		bool				resize;						///< Column header resizer is actively being dragged via mouse cursor
		bool				move;						///< Column is being moved via mouse cursor
	} mColumnDND;										///< Column drag and drop management
	struct
	{
		tColItem			*column;
		tRowItem			*row;
		Point2I				pos;						///< drawing start position
		bool				isDraw;						///< Whether or not we're performing the deferred draw at this moment
	} mDeferred;										///< Deferred drawing of a column and/or row

	bool cellSelected(Point2I cell);
	void onCellSelected(Point2I cell);

public:
	GuiModernTextListCtrl();
	~GuiModernTextListCtrl();

	DECLARE_CONOBJECT(GuiModernTextListCtrl);
#if !defined(SUPPORT_TGEA)
	DECLARE_CATEGORY("Gui Lists");
	DECLARE_DESCRIPTION("A control that has header columns and content rows that are tab delimited.\nLooks similar to a report view in Microsoft's ListView control.");
#endif // !SUPPORT_TGEA

	// control callbacks
	DECLARE_CALLBACK(void, onColumnMoved, (S32 colKey, U32 colOldPos, U32 colNewPos));
	DECLARE_CALLBACK(void, onColumnResized, (S32 colKey, U32 colPos, U32 width));
	DECLARE_CALLBACK(void, onColumnSelect, (S32 colKey, U32 colPos, S32 sortOrder, bool doubleClick));
	DECLARE_CALLBACK(void, onRowSelect, (S32 rowKey, U32 rowPos, S32 colKey, U32 colPos, bool doubleClick));
	DECLARE_CALLBACK(void, onRowUnselect, ());
	DECLARE_CALLBACK(void, onDeleteKey, (U32 index, S32 key));

	// control management
	static void initPersistFields();

	bool onAdd();
	void onRemove();
	bool onWake();
	void onSleep();
	bool onKeyDown(const GuiEvent &event);
	virtual void onRenderColumnHeaders(Point2I offset, Point2I parentOffset, Point2I headerDim);
	virtual void onRenderCell(Point2I offset, Point2I cell, bool selected, bool mouseOver);
	virtual void onRender(Point2I offset, const RectI &updateRect);

	void scrollCellVisible(Point2I cell);
	virtual void onMouseDown(const GuiEvent &event);
	virtual void onMouseUp(const GuiEvent &event);
	virtual void onMouseDragged(const GuiEvent &event);
	virtual void onMouseLeave(const GuiEvent &event);
	void checkColumnResizers(const GuiEvent &lastGuiEvent);
	virtual void getCursor(GuiCursor *&cursor, bool &showCursor, const GuiEvent &lastGuiEvent);


	// column header datablock handling
	void setHeaderProfile(GuiControlProfile *prof);
	static bool setHeaderProfileProt(void *object, const char *data);
	// column sort indicator datablock handling
	void setSortProfile(GuiControlProfile *prof);
	static bool setSortProfileProt(void *object, const char *data);

	/// Called when the mHeaderProfile is deleted
	virtual void onDeleteNotify(SimObject *object);


	// console exposed control methods
	U32 addColumn(S32 key, const char *name, U32 width, U32 minWidth, U32 maxWidth, const char *flags);
	U32 addRow(S32 key, const char *text, U32 index = -1, const char *flags = "");

	void removeRow(U32 index);

	void clearColumns(void);
	void clearRows(void);
	void clearSortOrders(void);

	void getColumnFlags(U32 index, char *flags);
	const char *getColumnName(U32 index);
	S32 getColumnKey(U32 index);
	U32 getColumnWidth(U32 index);
	U32 getColumnIndex(S32 key);
	U32 getColumnCount(void);

	U32 getSortOrderCount(void);
	const GuiModernTextListCtrl::tSortRule *getSortOrder(U32 index);

	U32 getSelectedRow(void);
	U32 getRowCount(void);
	void getRowFlags(U32 index, char *flags);
	const char *getRowText(U32 index);
	S32 getRowKey(U32 index);
	U32 getRowIndex(S32 key);

	void scrollVisible(U32 index);

	void setSelectedRow(U32 index);
	void setSortOrder(const char *rule);
	void setColumnFlags(U32 index, const char *flags);
	void setColumnSortOrder(S32 key, S32 order, bool pushToFirst = false);
	void setColumnWidth(U32 index, U32 width);
	void setRowFlags(U32 index, const char *flags);
	void setRowText(U32 index, const char *text);

	void sortColumns(void);
	void moveColumn(S32 key, U32 index);
	void moveRow(S32 key, U32 index);

	inline void stripMarkup(char *dest, const char *src) { stripMarkupText(dest, src, false); }
	inline void updateSize(void) { setSize(Point2I(0, 0)); }


	//...

	virtual void setCellSize( const Point2I &size ){ mCellSize = size; }
	virtual void getCellSize(       Point2I &size ){ size = mCellSize; }

	const char *getScriptValue();
	void setScriptValue(const char *value);

	void setSize(Point2I newSize);
	
};


//=============================================================================
// GuiModernTextListCtrl column sort class
//=============================================================================
class GuiModernSort
{
private:
	GuiModernTextListCtrl							*mOwner;
	std::vector<GuiModernTextListCtrl::tColItem>	*mColumns;
	std::vector<GuiModernTextListCtrl::tSortRule>	*mRules;
	bool											mMarkup;
	char											mTextA[MAXTEXTLEN];
	char											mTextB[MAXTEXTLEN];
#if !defined(MODERNLIST_SORT_STRIPONCE)
	char											mTextT[MAXTEXTLEN];
#endif // !MODERNLIST_SORT_STRIPONCE

public:
	GuiModernSort(GuiModernTextListCtrl *owner, std::vector<GuiModernTextListCtrl::tColItem> *columns, std::vector<GuiModernTextListCtrl::tSortRule> *rules)
	{
		mOwner		= owner;
		mColumns	= columns;
		mRules		= rules;
	}

	void setMarkupEnable(bool enable) { mMarkup = enable; }

	bool operator()(const GuiModernTextListCtrl::tRowItem *a, const GuiModernTextListCtrl::tRowItem *b);
/*	friend void swap(GuiModernTextListCtrl::tRowItem *&a, GuiModernTextListCtrl::tRowItem *&b)
	{
		GuiModernTextListCtrl::tRowItem *c;
		c = a;
		a = b;
		b = c;
	}
*/
};



#endif //_GUIMODERNTEXTLISTCTRL_H_
