<?php
/**************************************************************************
 *  Torque Server Query PHP Script
 *  Copyright (C) 2011-2012 by Nathan Martin  All Rights Reserved
 *
 *  Author can be contacted at nmartin <AT> gmail [DOT] com
 *  This source code is licensed under the General Public License (GPL) v2
 *  License info: http://www.gnu.org/copyleft/gpl.html
 *
 *=========================================================================
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *************************************************************************/

// global variables
$socket		= 0;
$servers	= array();
$response	= array('master' => false, 'pinged' => 0, 'reping' => false);

$stats['new']				= 0;	// new servers added to servers table.
$stats['updated']			= 0;	// updated servers in servers table.
$stats['newHistory']		= 0;	// new server history added to serversHistory table.
$stats['updatedHistory']	= 0;	// updated existing server history in serversHistory table.
$stats['retries']			= 0;	// number of servers we retried contacting for game ping response.
$stats['infoRetries']		= 0;	// number of servers we retried contacting for game info response.
$stats['different']			= 0;	// number of servers that were different from previous check
$stats['newHost']			= 0;	// number of new hosts
$stats['updatedHost']		= 0;	// number of updated hosts
$stats['active']			= 0;	// number of active servers that responded to game pings


$setup['db_host']			= 'localhost';
$setup['db_user']			= 'tgeMasterBrowser';
$setup['db_pass']			= 'FullPassGoesHere';
$setup['db_base']			= 'tgeMasterBrowser';

$setup['maxTries']			= 5;		// maximum number of tries on attempting to receive a server response


require("adodb5/adodb-exceptions.inc.php"); 
require('adodb5/adodb.inc.php');


// don't allow use through web server
if(isset($_SERVER['REMOTE_ADDR'])
{
	printf("Go Away!\n");
	exit(0);
}


//error_reporting(E_ERROR | E_PARSE | E_NOTICE);
error_reporting(E_ALL);


//=============================================================================
// Database Management
//=============================================================================

function DBConnect()
{
	global $DBC, $setup;

	try
	{
		$DBC = &ADONewConnection('mysql');
		$DBC->PConnect($setup['db_host'], $setup['db_user'], $setup['db_pass'], $setup['db_base']);
		$DBC->SetFetchMode(ADODB_FETCH_ASSOC);
//		$ADODB_FETCH_MODE = ADODB_FETCH_ASSOC;

	} catch (exception $e)
	{ 
		var_dump($e); 
		adodb_backtrace($e->gettrace());

		return false;
	}

	return true;
}

function DBHandleGameType(&$server)
{
	global $DBC, $stats;

	try
	{
		// see if game type exists in our games table
		$sql = "SELECT * FROM `games` WHERE `gameName` = ? LIMIT 1;";
		$res = $DBC->Execute($sql, array($server['gameType']));

		if($res->fields)
		{
			$ID = $res->fields['gameID'];

			// successfully got ID, now update last seen timestamp
			$sql = "UPDATE `games` SET `tsLastSeen`=CURRENT_TIMESTAMP() WHERE `gameID`='". $ID ."';";
			$DBC->Execute($sql);

			$server['gameType'] = $ID;

			// return host ID
			return $ID;
		}

		// game type isn't in our database, insert it
		$sql = "INSERT INTO `games` (gameName, tsFirstSeen) VALUES (?, CURRENT_TIMESTAMP);";
		$res = $DBC->Execute($sql, array($server['gameType']));
		$ID  = $DBC->Insert_ID();

		$server['gameType'] = $ID;

		// return host ID
		return $ID;

	} catch (exception $e)
	{
		var_dump($e);
		adodb_backtrace($e->gettrace());

		return false;
	}
}

function DBHandleMissionType(&$server)
{
	global $DBC, $stats;

	try
	{
		// see if mission type exists in our missions table
		$sql = "SELECT * FROM `missions` WHERE `missionName` = ? LIMIT 1;";
		$res = $DBC->Execute($sql, array($server['missionType']));

		if($res->fields)
		{
			$ID = $res->fields['missionID'];

			// successfully got ID, now update last seen timestamp
			$sql = "UPDATE `missions` SET `tsLastSeen`=CURRENT_TIMESTAMP() WHERE `missionID`='". $ID ."';";
			$DBC->Execute($sql);

			$server['missionType'] = $ID;

			// return host ID
			return $ID;
		}

		// mission type isn't in our database, insert it
		$sql = "INSERT INTO `missions` (missionName, tsFirstSeen) VALUES (?, CURRENT_TIMESTAMP);";
		$res = $DBC->Execute($sql, array($server['missionType']));
		$ID  = $DBC->Insert_ID();

		$server['missionType'] = $ID;

		// return host ID
		return $ID;

	} catch (exception $e)
	{
		var_dump($e);
		adodb_backtrace($e->gettrace());

		return false;
	}
}


function DBServerInfoToDBFields(&$server, $forHistory = false)
{
	$fields = array();

//	$fields['']		= $server["address"];
//	$fields['']		= $server["lastSeen"];
//	$fields['']		= $server["queryVersion"];
//	$fields['']		= $server["netCurrentVersion"];
//	$fields['']		= $server["netMinimumVersion"];

	if(!$forHistory)
	{
		$fields['hostID']	= $server["hostID"];
		$fields['port']		= $server["port"];
	}
	else
	{
		$fields['serverID']	= $server["serverID"];
	}

	$fields['gameType']		= $server["gameType"];
	$fields['missionType']	= $server["missionType"];
	$fields['status']		= $server["status"];
	$fields['playerCount']	= $server["playerCount"];
	$fields['playerMax']	= $server["playerMax"];
	$fields['botCount']		= $server["botCount"];
	$fields['cpuSpeed']		= $server["CPUSpeed"];
	$fields['gameVersion']	= $server["gameVersion"];
	$fields['name']			= $server["name"];
	$fields['missionName']	= $server["missionName"];
	$fields['info']			= $server["info"];
	$fields['infoLong']		= $server["infoLong"];

	return $fields;
}

function DBProcAndGetHostID(&$server)
{
	global $DBC, $stats;

	try
	{
		// see if host exists in our host table
		$sql = "SELECT * FROM `hosts` WHERE `hostAddress` = ? LIMIT 1;";
		$res = $DBC->Execute($sql, array($server['address']));

		if($res->fields)
		{
			$hostID = $res->fields['hostID'];

			// successfully got ID, now update last seen timestamp
			$sql = "UPDATE `hosts` SET `tsLastSeen`=CURRENT_TIMESTAMP() WHERE `hostID`='". $hostID ."';";
			$DBC->Execute($sql);

			$server['hostID'] = $hostID;

			$stats['updatedHost']++;

			// return host ID
			return $hostID;
		}

		// host isn't in our database, insert it
		$sql = "INSERT INTO `hosts` (hostAddress, tsFirstSeen) VALUES (?, CURRENT_TIMESTAMP);";
		$res = $DBC->Execute($sql, array($server['address']));
		$hostID = $DBC->Insert_ID();

		$server['hostID'] = $hostID;

		$stats['newHost']++;

		// return host ID
		return $hostID;

	} catch (exception $e)
	{
		var_dump($e);
		adodb_backtrace($e->gettrace());

		return false;
	}
}

function DBProcAndGetServerID(&$server)
{
	global $DBC, $stats;
	$hostID = $server['hostID'];

	try
	{
		// see if server exists in our servers table
		$sql = "SELECT * FROM `servers` WHERE `hostID` = ? AND `port` = ? LIMIT 1;";
		$res = $DBC->Execute($sql, array($hostID, $server['port']));

		if($res->fields)
		{
			$serverID = $res->fields['serverID'];

			// successfully got ID, now update last seen timestamp
			$sql = "UPDATE `servers` SET `tsLastSeen`=CURRENT_TIMESTAMP WHERE `serverID`='". $serverID ."';";
			$DBC->Execute($sql);

			$server['serverID']	= $serverID;
			$server['isDiff']	= DBCheckServerDifferent($server, $res->fields);

			// return host ID
			return $serverID;
		}

		// server isn't in our database, insert it
		$fields = DBServerInfoToDBFields($server);

		$rs  = 'servers';
		$sql = $DBC->GetInsertSQL($rs, $fields);
		$res = $DBC->Execute($sql);
		$serverID = $DBC->Insert_ID();

		// successfully got ID, now update first and last seen timestamps,
		$sql = "UPDATE `servers` SET `tsFirstSeen`=CURRENT_TIMESTAMP, `tsLastSeen`=CURRENT_TIMESTAMP WHERE `serverID`='". $serverID ."';";
		$DBC->Execute($sql);

		$server['serverID'] = $serverID;

		$stats['new']++;

		// return server ID
		return $serverID;

	} catch (exception $e)
	{
		var_dump($e);
		adodb_backtrace($e->gettrace());

		return false;
	}
}

function DBCheckServerDifferent(&$server, $fields)
{
	global $stats;


	$nfields = DBServerInfoToDBFields($server);
	unset($fields['serverID'], $fields['tsFirstSeen'], $fields['tsLastSeen']);

	// see if there's any differences
	foreach($fields as $key => $col)
	{
		if($nfields[$key] != $fields[$key])
		{
//			printf("%s is new:\n  Old: %s\n  New: %s\n",
//					$key, $fields[$key], $nfields[$key]);
			$stats['different']++;
			return true;
		}
	}

	return false;
}

function DBUpdateServerAndHistory(&$server)
{
	global $DBC, $stats;

	// oldest a history record will be since last update before it is abandoned for being too old?
	$maxAge = 10 * 60; // 10 minutes


	try
	{
		if(!isset($server['isDiff']))
			$server['isDiff'] = true;

		// has the server info changed since last recording (or is it new)?
		if($server['isDiff'] == true)
		{
			// yes, update the server record
			$fields = DBServerInfoToDBFields($server, false);
			$DBC->AutoExecute('servers', $fields, 'UPDATE', 'serverID = \''. $server['serverID'] .'\'', true);

//			$sql = "UPDATE `servers` SET `tsLastSeen`=CURRENT_TIMESTAMP WHERE `serverID`='". $server['serverID'] ."';";
//			$DBC->Execute($sql);

			$stats['updated']++;

NewHistoryRecord:
			// insert new server history record
			$fields = DBServerInfoToDBFields($server, true);
			$rs  = 'serversHistory';
			$sql = $DBC->GetInsertSQL($rs, $fields);
			$res = $DBC->Execute($sql);
			$historyID = $DBC->Insert_ID();

			$sql = "UPDATE `serversHistory` SET `tsRecorded`=CURRENT_TIMESTAMP, `tsLastMatched`=CURRENT_TIMESTAMP WHERE `serverHistoryID`='". $historyID ."';";
			$DBC->Execute($sql);

			$stats['newHistory']++;

			// done, return current history ID
			return $historyID;
		}

		// get the last history entry for this server
		$sql = "SELECT *, UNIX_TIMESTAMP(`tsLastMatched`) as `lastSeen` FROM `serversHistory` WHERE `serverID` = ? AND `tsLastMatched` >= FROM_UNIXTIME(?) ORDER BY `serverHistoryID` DESC LIMIT 1;";
//		$sql = "SELECT *, UNIX_TIMESTAMP(`tsLastMatched`) as `lastSeen` FROM `serversHistory` WHERE serverID = ? AND lastSeen >= ? ORDER BY `serverHistoryID` DESC LIMIT 1;";
		$res = $DBC->Execute($sql, array($server['serverID'], time() - $maxAge));

		if(!$res->fields)
			goto NewHistoryRecord;

		$historyID = $res->fields['serverHistoryID'];

		// update last match/seen history
		$sql = "UPDATE `serversHistory` SET `tsLastMatched`=CURRENT_TIMESTAMP WHERE `serverHistoryID`='". $historyID ."';";
		$DBC->Execute($sql);

		$stats['updatedHistory']++;

		// done, return current history ID
		return $historyID;

	} catch (exception $e)
	{
		var_dump($e);
		adodb_backtrace($e->gettrace());

		return false;
	}
}

function DBUpdateStats()
{
	global $DBC, $stats;

	try
	{
		$fields['activeServers']	= $stats['active'];			// number of active servers that responded to game pings.
		$fields['newServers']		= $stats['new'];			// new servers added to servers table.
		$fields['updatedServers']	= $stats['updated'];		// updated servers in servers table.
		$fields['newHistory']		= $stats['newHistory'];		// new server history added to serversHistory table.
		$fields['updatedHistory']	= $stats['updatedHistory'];	// updated existing server history in serversHistory table.
		$fields['serversRetried']	= $stats['retries'];		// number of servers we retried contacting for game ping response.
		$fields['serversDifferent']	= $stats['different'];		// number of servers that were different from previous check
		$fields['newHosts']			= $stats['newHost'];		// number of new hosts
		$fields['updatedHosts']		= $stats['updatedHost'];	// number of updated hosts

		$rs  = 'masterStats';
		$sql = $DBC->GetInsertSQL($rs, $fields);
		$res = $DBC->Execute($sql);
		$ID = $DBC->Insert_ID();

		// return stats ID
		return $ID;

	} catch (exception $e)
	{
		var_dump($e);
		adodb_backtrace($e->gettrace());

		return false;
	}
}



function DBProcServer(&$server)
{
	$gameID		= DBHandleGameType($server);
	$missionID	= DBHandleMissionType($server);
	$hostID		= DBProcAndGetHostID($server);
	$serverID	= DBProcAndGetServerID($server);

	DBUpdateServerAndHistory($server);

/*
	$server["address"];
	$server["port"];
	$server["lastSeen"];
	$server["queryVersion"];
	$server["netCurrentVersion"];
	$server["netMinimumVersion"];
	$server["gameVersion"];
	$server["name"];
	$server["gameType"];
	$server["missionType"];
	$server["missionName"];
	$server["status"];
	$server["playerCount"];
	$server["playerMax"];
	$server["botCount"];
	$server["CPUSpeed"];
	$server["info"];
	$server["infoLong"];
*/
}



//=============================================================================
// Utility Functions
//=============================================================================

/**
 * View any string as a hexdump.
 *
 * This is most commonly used to view binary data from streams
 * or sockets while debugging, but can be used to view any string
 * with non-viewable characters.
 *
 * @version     1.3.2
 * @author      Aidan Lister <aidan@php.net>
 * @author      Peter Waller <iridum@php.net>
 * @link        http://aidanlister.com/2004/04/viewing-binary-data-as-a-hexdump-in-php/
 * @param       string  $data        The string to be dumped
 * @param       bool    $htmloutput  Set to false for non-HTML output
 * @param       bool    $uppercase   Set to true for uppercase hex
 * @param       bool    $return      Set to true to return the dump
 */
function hexdump ($data, $htmloutput = true, $uppercase = false, $return = false)
{
    // Init
    $hexi   = '';
    $ascii  = '';
    $dump   = ($htmloutput === true) ? '<pre>' : '';
    $offset = 0;
    $len    = strlen($data);

    // Upper or lower case hexadecimal
    $x = ($uppercase === false) ? 'x' : 'X';

    // Iterate string
    for ($i = $j = 0; $i < $len; $i++)
    {
        // Convert to hexidecimal
        $hexi .= sprintf("%02$x ", ord($data[$i]));

        // Replace non-viewable bytes with '.'
        if (ord($data[$i]) >= 32) {
            $ascii .= ($htmloutput === true) ?
                            htmlentities($data[$i]) :
                            $data[$i];
        } else {
            $ascii .= '.';
        }

        // Add extra column spacing
        if ($j === 7) {
            $hexi  .= ' ';
            $ascii .= ' ';
        }

        // Add row
        if (++$j === 16 || $i === $len - 1) {
            // Join the hexi / ascii output
            $dump .= sprintf("%04$x  %-49s  %s", $offset, $hexi, $ascii);

            // Reset vars
            $hexi   = $ascii = '';
            $offset += 16;
            $j      = 0;

            // Add newline
            if ($i !== $len - 1) {
                $dump .= "\n";
            }
        }
    }

    // Finish dump
    $dump .= $htmloutput === true ?
                '</pre>' :
                '';
    $dump .= "\n";

    // Output method
    if ($return === false) {
        echo $dump;
    } else {
        return $dump;
    }
}

function flushnow()
{
    echo(str_repeat(' ',256));
    // check that buffer is actually set before flushing
    if (ob_get_length()){           
        @ob_flush();
        @flush();
        @ob_end_flush();
    }   
    @ob_start();
}

function isPrintableString($str)
{
	$len = strlen($str);

	for($i=0; $i<$len; $i++)
	{
		$value = ord(substr($str, $i, 1));
		if($value < 0x20 || $value > 0x7E)
			return false;
	}

	return true;
}

function getTimeMinsSecs($secs)
{
	// get minutes
	$mins  = intval($secs / 60);
	$secs -= $mins * 60;

	return sprintf('%02u:%02u', $mins, $secs);
}

function printLog($text)
{
	global $setup;


	if(!isset($setup['startTS']))
	{
		$setup['startTS'] = time();

		printf("\n---------------------------------------------------------------------\n".
				" Torque Masterd Query started on %s\n".
				"---------------------------------------------------------------------\n",
				date('Y-m-d H:i:s (P \UTC [T])', $setup['startTS'])
				);
	}

	printf("[%s] %s",
			getTimeMinsSecs(time() - $setup['startTS']),
			$text);
}


//=============================================================================
// Packet Management
//=============================================================================

function writeU64(&$data, $num)
{
	//$data .= pack('C', $num);
}

function writeU32(&$data, $num)
{
	$data .= pack('V', $num);
}

function writeU16(&$data, $num)
{
	$data .= pack('v', $num);
}

function writeU8(&$data, $num)
{
	$data .= pack('C', $num);
}

function readU64(&$data)
{
	list(, $lolo, $lohi, $hilo, $hihi) = unpack('v*', substr($data, 0, 8));
	$data = substr($data, 8);

	return ($hihi * (0xffff+1) + $hilo) * (0xffffffff+1) + ($lohi * (0xffff+1) + $lolo);
}

function readU32(&$data)
{
	list(,$val) = unpack('V', substr($data, 0, 4));
	$data       = substr($data, 4);

	return $val;
}

function readU16(&$data)
{
	list(,$val) = unpack('v', substr($data, 0, 2));
	$data       = substr($data, 2);

	return $val;
}

function readChar(&$data)
{
	$val  = substr($data, 0, 1);
	$data = substr($data, 1);

	return $val;
}

function readU8(&$data)
{
	return ord(readChar($data));
}

function readNULLString(&$data)
{
	$str = '';

	if( ($x = strpos($data, "\x00")) !== false)
	{
		$str  = substr($data, 0, $x);
		$data = substr($data, $x +1);
	}

	return $str;
}

function writeCString(&$data, $str)
{
	$len = 0;

	// get string length
	$len = strlen($str);

	// clamp the string to 255 bytes
	if($len > 0xFF)
		$len = 0xFF;

	// write string length
	writeU8($data, $len);

	// write string content
	$data .= substr($str, 0, $len);
}

function readCString(&$data)
{
	$str = '';
	$len = 0;

	// read string length
	$len = readU8($data);

	// abort on no length
	if(!$len)
		return $str;

	// read in string
	$str  = substr($data, 0, $len);
	$data = substr($data, $len);

	return $str;
}

function readLongCString(&$data)
{
	$str = '';
	$len = 0;

	// read string length
	$len = readU16($data);

	// abort on no length
	if(!$len)
		return $str;

	// read in string
	$str  = substr($data, 0, $len);
	$data = substr($data, $len);

	return $str;
}


// Tribes 2 / Torque Query and Response Types
define("MasterServerGameTypesRequest",	2);		// *
define("MasterServerGameTypesResponse",	4);		// !
define("MasterServerListRequest",		6);		// *
define("MasterServerListResponse",		8);		// !
define("GameMasterInfoRequest",			10);	// !
define("GameMasterInfoResponse",		12);	// *
define("GamePingRequest",				14);
define("GamePingResponse",				16);
define("GameInfoRequest",				18);
define("GameInfoResponse",				20);
define("GameHeartbeat",					22);	// *
define("MasterServerInfoRequest",		24);	// *, Torque doesn't use this...
define("MasterServerInfoResponse",		26);

// Packet [Query] Flags
define("OfflineQuery",					0x01);	// when not set, it's a OnlineQuery
define("NoStringCompress",				0x02);	// Do not compress strings


//=========================================================================
// Packet Header Management
//=========================================================================
function createHeader($type, $flags, $session, $key)
{
	$header = array();

	$header['type']		= $type;
	$header['flags']	= $flags;
	$header['session']	= $session;
	$header['key']		= $key;

	return $header;
}

function writeHeader(&$data, &$header)
{
	writeU8( $data, $header['type']);
	writeU8( $data, $header['flags']);
	writeU16($data, $header['session']);
	writeU16($data, $header['key']);
}

function readHeader(&$data)
{
	$type		= readU8($data);
	$flags		= readU8($data);
	$session	= readU16($data);
	$key		= readU16($data);

	return createHeader($type, $flags, $session, $key);
}


//=========================================================================
// Game Server Lists
//=========================================================================
function createMasterListQuery($game, $mission)
{
	$data = '';

	// session and key are just random numbers, so 12 and 34 will do
	$header = createHeader(MasterServerListRequest, 0x00, 12, 34);

	writeHeader( $data, $header);		// header
	writeU8(     $data, 0xFF);			// packet index - 0xFF is query, other resend packet
	writeCString($data, $game);			// game type
	writeCString($data, $mission);		// mission type
	writeU8(     $data, 0);				// min players
	writeU8(     $data, 0);				// max players
	writeU32(    $data, 0xFFFFFFFF);	// region mask  - any region
	writeU32(    $data, 0);				// min version  - any version
	writeU8(     $data, 0);				// info flags   - any server status
	writeU8(     $data, 0);				// max bots     - any bots
	writeU16(    $data, 0);				// min CPU      - any processor speed
	writeU8(     $data, 0);				// buddy count  - no buddies to provide

	return $data;
}

function parseMasterListResponse(&$data)
{
	global $servers;
	$result		= array();
	$IPv4		= array();
//	$servers	= array();
	$port		= 0;
	$num		= 0;


	// set to server count, easier to debug if its first entry
	if(!isset($servers['count']))
		$servers['count'] = 0;

	// read packet index and total packets
	$pindex = readU8($data);	// Packet Index
	$ptotal = readU8($data);	// Packet Total

	// read number of servers in this packet
	$num = readU16($data);

	// report master server response
	printLog(sprintf("List Packet Received: [I:%u, T:%u, C:%u]\n", $pindex, $ptotal, $num));


	// populate the servers records
	for($i=0; $i<$num; $i++)
	{
		// read in IP address as four bytes
		for($n=0; $n<4; $n++)
			$IPv4[$n] = readU8($data);

		// read in port number
		$port = readU16($data);

		$servers[$i]['address']		= sprintf("%u.%u.%u.%u", $IPv4[0], $IPv4[1], $IPv4[2], $IPv4[3]);
		$servers[$i]['port']		= $port;
		$servers[$i]['lastSeen']	= 0;
		$servers[$i]['stage']		= 0;
		$servers[$i]['tried']		= 0;
	}

	// update server count and servers array.
	$servers['count']	+= $num;
	$result[ 'servers']	 = $servers;

	// results
	return $result;
}

function procMasterListResponse(&$remoteHost, &$result)
{
	global $servers, $response;


	// ping all the servers
	$num  = $servers['count'];
	$dout = createGamePingQuery();

	for($i=0; $i<$num; $i++)
	{
		$server = &$servers[$i];
		socketSend($server['address'], $server['port'], $dout);

		printLog(sprintf("Sent GamePing query to %s:%lu\n",
							$server['address'], $server['port']));
	}

	// remember when we pinged all servers
	$response['master'] = true;
	$response['pinged'] = time();

	// done
}

function repingServers()
{
	global $servers, $stats, $setup;


	// retry contacting all non-responsive servers, just in case the previous request packet was lost
	$num  = $servers['count'];

	for($i=0; $i<$num; $i++)
	{
		$server = &$servers[$i];

		if( ($server['stage'] < 2) &&
			($server['tried'] < $setup['maxTries']) &&
			((time() - $server['lastSeen']) > 2))
		{
			switch ($server['stage'])
			{
				default:
				case 0: // send game ping request
				{
					$dout = createGamePingQuery();
					$stats['retries']++;
					break;
				}

				case 1: // send game info request
				{
					$dout = createGameInfoQuery();
					$stats['infoRetries']++;
					break;
				}
			}

			$server['tried']++;
			socketSend($server['address'], $server['port'], $dout);

			$queryType = ($server['stage'] == 1)? 'GameInfo':'GamePing';
			printLog(sprintf("Resent %s query to %s:%lu\n",
								$queryType, $server['address'], $server['port']));
		}
	}

	// done
}

function getNumServersAtStage($stage = 2)
{
	global $servers, $stats;


	// count the number of servers at specified stage
	$num  = $servers['count'];

	$counter = 0;
	for($i=0; $i<$num; $i++)
	{
		$server = &$servers[$i];

		if($server['stage'] == $stage)
			$counter++;
	}

	// done
	return $counter;
}

function &getServerRecord(&$remoteHost)
{
	global $servers;


	// locate the server record associated with the remote address:port
	$num = $servers['count'];
	for($i=0; $i<$num; $i++)
	{
		$server = &$servers[$i];
		if( $server['port'] == $remoteHost['port'] &&
			!strcasecmp($server['address'], $remoteHost['address']))
		{
			return $servers[$i];
		}
	}

	return false;
}


//=========================================================================
// Game Master Server Info
//=========================================================================

function createGameMasterInfoQuery($game, $mission)
{
	$data = '';

	// session and key are just random numbers, so 12 and 34 will do
	$header = createHeader(GameMasterInfoRequest, 0x00, 12, 34);

	return $data;
}

function parseGameMasterInfoResponse(&$data)
{
	$result		= array();
	$num		= 0;


	$result['gameType']			= strval(readCString($data));		// Game Type
	$result['missionType']		= strval(readCString($data));		// Mission Type
	$result['playerMax']		= readU8($data) + 0;				// Player Maximum
	$result['regionMask']		= readU32($data) + 0;				// Region mask
	$result['gameVersion']		= readU32($data) + 0;				// Game Version [integer]
	$result['status']			= readU8($data) + 0;				// Status bitflags
	$result['botCount']			= readU8($data) + 0;				// Bot Count
	$result['CPUSpeed']			= readU16($data) + 0;				// Processor Speed (MHz)
	$result['playerCount']		= readU8($data) + 0;				// Player Count

	// results
	return $result;
}

function procGameMasterInfoResponse(&$remoteHost, &$result)
{

	// remove unused/unsupported fields
	unset($result['regionMask']);

	// fill in the non-provided fields
	$result['info']				= '';	// Info Line
	$result['infoLong']			= '';	// Misc. Details
	$result['masterInfo']		= true;

	// let another function handle the rest
	procGameInfoResponse($remoteHost, $result);

	// done
}


//=========================================================================
// Misc. Master Server Information [Non-Standard]
//=========================================================================
function createMasterInfoQuery()
{
	$data = '';

	// session and key are just random numbers, so 0x8C and 0xA4 will do
	$header = createHeader(MasterServerInfoRequest, 0x00, 0x8C, 0xA4);

	writeHeader( $data, $header);		// header

	return $data;
}

//=========================================================================
// Game and Mission Types
//=========================================================================
function createMasterTypesQuery()
{
	$data = '';

	// session and key are just random numbers, so 0x8C and 0xA4 will do
	$header = createHeader(MasterServerGameTypesRequest, 0x00, 0x8C, 0xA4);

	writeHeader( $data, $header);		// header

	return $data;
}

function parseMasterTypesResponse(&$data)
{
	$result		= array();
	$games		= array();
	$missions	= array();
	$num		= 0;

	// read game types
	$num = readU8($data);
	for($i=0; $i<$num; $i++)
		$games[$i] = readCString($data);

	// read mission types
	$num = readU8($data);
	for($i=0; $i<$num; $i++)
		$missions[$i] = readCString($data);

	$result['gameTypes']	= $games;
	$result['missionTypes']	= $missions;
	
	// results
	return $result;
}


//=========================================================================
// Game Ping
//=========================================================================
function createGamePingQuery()
{
	$data = '';

	// session and key are just random numbers, so 0x8C and 0xA4 will do
	$header = createHeader(GamePingRequest, NoStringCompress, 0x8C, 0xA4);

	writeHeader( $data, $header);		// header

	return $data;
}

function parseGamePingResponse(&$data)
{
	global $stats;
	$result		= array();
	$num		= 0;

	$result['queryVersion']			= strval(readCString($data));	// Version string -- "This is basically the server query protocol version now"
	$result['netCurrentVersion']	= readU32($data) + 0;			// Network Protocol Version         (GameConnection::CurrentProtocolVersion)
	$result['netMinimumVersion']	= readU32($data) + 0;			// Network Minimum Protocol Version (GameConnection::MinRequiredProtocolVersion)
	$result['gameVersion']			= readU32($data) + 0;			// Game Version [integer]
	$result['name']					= strval(readCString($data));	// Game Server Name

	$stats['active']++;

	// results
	return $result;
}

function procGamePingResponse(&$remoteHost, &$result)
{
	// get the game server record
	$server = &getServerRecord($remoteHost);
	if($server === false)
		return;

	// ignore ping response if we've already received it before
	if($server['stage'] > 0)
		return;

	// merge the result into the record
	$server = array_merge($server, $result);

	// send game information query to server
	$dout = createGameInfoQuery();
	socketSend($remoteHost['address'], $remoteHost['port'], $dout);

	printLog(sprintf("Sent GameInfo query to %s:%lu\n",
						$server['address'], $server['port']));

	// mark the server as has responded
	$server['lastSeen'] = time();
	$server['stage']	= 1;

	// done
}


//=========================================================================
// Game Information
//=========================================================================
function createGameInfoQuery()
{
	$data = '';

	// session and key are just random numbers, so 0x8C and 0xA4 will do
	$header = createHeader(GameInfoRequest, NoStringCompress, 0x8C, 0xA4);

	writeHeader( $data, $header);		// header

	return $data;
}

function parseGameInfoResponse(&$data)
{
	$result		= array();
	$num		= 0;


	$result['gameType']			= strval(readCString($data));		// Game Type
	$result['missionType']		= strval(readCString($data));		// Mission Type
	$result['missionName']		= strval(readCString($data));		// Mission Name
	$result['status']			= readU8($data) + 0;				// Status bitflags
	$result['playerCount']		= readU8($data) + 0;				// Player Count
	$result['playerMax']		= readU8($data) + 0;				// Player Maximum
	$result['botCount']			= readU8($data) + 0;				// Bot Count
	$result['CPUSpeed']			= readU16($data) + 0;				// Processor Speed (MHz)
	$result['info']				= strval(readCString($data));		// Info Line
	$result['infoLong']			= strval(readLongCString($data));	// Misc. Details

	// validate the gameType
	if(!isPrintableString($result['gameType']))
	{
		// gameType is malformed, possibly response packet isn't Torque standard structured format.
		$result['isMalformed'] = true;
	}

	// results
	return $result;
}

function procGameInfoResponse(&$remoteHost, &$result)
{
	// check for malformed result
	if(isset($result['isMalformed']) && !isset($result['masterInfo']))
	{
		// gameinfo response was malformed, try master info as alternative.
		// send game information query to server
		$dout = createGameInfoQuery();
		socketSend($remoteHost['address'], $remoteHost['port'], $dout);

		return;
	}

	// get the game server record
	$server = &getServerRecord($remoteHost);
	if($server === false)
		return;

	// ignore info response if we've already received it before
	if($server['stage'] > 1)
		return;

	// merge the result into the record
	$server = array_merge($server, $result);

	// mark the server as has responded
	$server['lastSeen'] = time();
	$server['stage']	= 2;

	// DEBUG: show the results
//	echo("<pre>\n"); var_dump($server); echo("</pre>\n");

	// finally, pass on the server to the database manager
	DBProcServer($server);

	// done
}


//=========================================================================
// Socket Handling and Processing
//=========================================================================

function socketOpen()
{
	global $socket;

	if( ($socket = socket_create(AF_INET, SOCK_DGRAM, getprotobyname('udp'))) === FALSE )
	{
		print('Error: Could not create socket '. socket_last_error() .': '. socket_strerror(socket_last_error()) ."\n");
		return false;
	}
}

function socketClose()
{
	global $socket;

	socket_close($socket);
}

function socketSend($host, $port, &$data)
{
	global $socket;

	socket_sendto($socket, $data, strlen($data), 0x100, $host, $port);
}

function socketPoll($timeout = 200)
{
	global $socket;
	$rhost					= '';
	$rport					= '';
	$buffer					= '';
	$num_changed_sockets	= 0;
	$serviced				= 0;
	$read					= array($socket);

	$num_changed_sockets = socket_select($read, $write = NULL, $except = NULL, 0, $timeout * 1000);
	if($num_changed_sockets === false)
	{
		print 'socket_select() failed '. socket_last_error($socket) .':'. socket_strerror(socket_last_error($socket));
	} else if( ($num_changed_sockets > 0) && ($read[0] == $socket) )
	{
		while(($size = @socket_recvfrom($socket, $buffer, 4 * 1024, MSG_DONTWAIT, $rhost, $rport)))
		{
//			printf("<p>Packet Received from <b><u>%s</u>:<u>%lu</u></b><p>", $rhost, $rport);
			printLog(sprintf("Packet Received from %s:%lu\n", $rhost, $rport));
			socketPacket($rhost, $rport, $buffer);
			$serviced++;

//			printf("Socket Recv[%u] %u\n", $serviced, $size);
		}
	}

	// return number of sockets serviced
	return $serviced;
}

function socketPacket(&$rhost, &$rport, &$data)
{
	$din	= &$data;
	$dout	= '';


	$remoteHost['address']	= &$rhost;
	$remoteHost['port']		= &$rport;

	// read header
	$head = readHeader($din);

	// process the packet based on type
	switch($head['type'])
	{
		case MasterServerGameTypesResponse:
			$result = parseMasterTypesResponse($din);
			break;
		
		case MasterServerListResponse:
			$result = parseMasterListResponse($din);
			procMasterListResponse($remoteHost, $result);
			break;

		case GamePingResponse:
			$result = parseGamePingResponse($din);
			procGamePingResponse($remoteHost, $result);
			break;

		case GameInfoResponse:
			$result = parseGameInfoResponse($din);
			procGameInfoResponse($remoteHost, $result);
			break;

		case GameMasterInfoResponse:
			$result = parseGameMasterInfoResponse($din);
			procGameMasterInfoResponse($remoteHost, $result);
			break;
	}

}



function masterdClient(&$host, &$port)
{
	global $response, $stats;


	// set our process timeout to 10 seconds of no activity
	$timeout = 10;

	// get a socket to work with
	socketOpen();

	// send master server the list request query
	$dout = createMasterListQuery('Any', 'Any');
	socketSend($host, $port, $dout);

	// set last time we worked on a packet
	$lastWorked = time();

	// now keep waiting for and processing packets until times up
	while(true)
	{
		// get current timestamp in seconds
		$ts = time();

		// poll socket and then update last worked time if received anything
		if(socketPoll(200) > 0)
			$lastWorked = $ts;

		// check on the master reply
		if($response['master'] == false && (($ts - $lastWorked) > 3))
		{
			// no response in past 3 seconds, resend master server the list request query
			socketSend($host, $port, $dout);
		}

		// retry non-responded servers every second
		if($response['pinged'] != 0 && (($ts - $response['pinged']) >= 2))
		{
			$response['pinged'] = $ts;
			repingServers();
		}

		// exit polling loop after X seconds of no activity
		if(($ts - $lastWorked) > $timeout)
			break;
	}

	// done with the socket
	socketClose();

	// correct the number of active servers
	$stats['active'] = getNumServersAtStage(2);

	// we're done here
	return 0;
}


// set the master server address
//$host = "master.dottools.net:28002";
$host = "master.garagegames.com:28002";

$port = 28002;
if( ($x = strpos($host, ':')) !== false)
{
	$port = substr($host, $x +1);
	$host = substr($host, 0, $x);
}

// establish a connection with the database first
if(!DBConnect())
{
	printLog("Connection to Database failed!\n");
	exit(-1);
}

// initiat the master server client process
masterdClient($host, $port);

// store stats/results into database
DBUpdateStats();

// report results
printLog(
sprintf( "--- Results ---\n".
		"%u new and %u updated hosts\n".
		"%u new and %u updated (diff %u) servers\n".
		"%u new and %u updated servers' history\n",
		$stats['newHost'], $stats['updatedHost'],
		$stats['new'], $stats['updated'], $stats['different'],
		$stats['newHistory'], $stats['updatedHistory']
)
);

/****
NOTES:

	This query saved several rows' bacon:
	UPDATE `masterStats` SET newHosts = updatedHosts, updatedHosts = @temp WHERE (@temp := newHosts) IS NOT NULL AND tsRecorded <= '2011-06-20 17:35:09';

	Accidently had newHosts and updatedHosts keys in $stats backwards.
****/


/*
// testing.....
torqueQuery($host, $port, GamePingRequest);
torqueQuery($host, $port, GameInfoRequest);
*/

// all done
exit(0);

?>
